package ru.yandex.mail.so.factors.eml2html;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.james.mime4j.MimeException;

import ru.yandex.charset.Decoder;
import ru.yandex.function.EmptyRunnable;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.NullConsumer;
import ru.yandex.function.Processable;
import ru.yandex.io.ByteArrayInputStreamFactory;
import ru.yandex.sanitizer2.HtmlNode;
import ru.yandex.sanitizer2.PageHeaderException;
import ru.yandex.sanitizer2.SanitizingHandler;
import ru.yandex.util.storage.MailTextExtractor;
import ru.yandex.util.storage.StorageData;

public class Eml2Html implements GenericAutoCloseable<RuntimeException> {
    public static final Charset DEFAULT_EMAIL_CHARSET =
        Charset.forName("KOI8-R");

    private final ImmutableEml2HtmlConfig config;
    private final SanitizingHandler sanitizer;

    public Eml2Html(final ImmutableEml2HtmlConfig config)
        throws PageHeaderException
    {
        this.config = config;
        sanitizer = new SanitizingHandler(
            config.sanitizingConfig(),
            NullConsumer.instance(),
            EmptyRunnable.INSTANCE);
    }

    @Override
    public void close() {
        sanitizer.close();
    }

    @Nullable
    public HtmlNode convert(
        @Nonnull final Processable<byte[]> input)
        throws IOException, MimeException
    {
        StorageData rawHtml = MailTextExtractor.INSTANCE.extractDataFromEml(
            input.processWith(ByteArrayInputStreamFactory.INSTANCE),
            config);
        if (rawHtml == null || !rawHtml.metainfo().startsWith("text/html")) {
            return null;
        }

        Charset htmlCharset = rawHtml.contentType().getCharset();
        if (htmlCharset == null) {
            htmlCharset = DEFAULT_EMAIL_CHARSET;
        }
        Decoder decoder = new Decoder(
            htmlCharset.newDecoder()
                .onMalformedInput(CodingErrorAction.REPLACE)
                .onUnmappableCharacter(CodingErrorAction.REPLACE));
        rawHtml.processWith(decoder);
        rawHtml = null;
        HtmlNode root = sanitizer.sanitize(decoder, false);
        decoder = null;
        BlockquoteCountingVisitor blockquoteCounter =
            new BlockquoteCountingVisitor();
        root.accept(blockquoteCounter);
        if (blockquoteCounter.count() == 1) {
            // Single blockquote will be folded, so remove it
            root = root.accept(BlockquoteErasingVisitor.INSTANCE);
        }
        return root;
    }
}
