package ru.yandex.mail.so.factors.eml2html;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;
import org.apache.james.mime4j.MimeException;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.HtmlSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.sanitizer2.HtmlNode;
import ru.yandex.sanitizer2.PageHeaderException;

public class Eml2HtmlExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(BinarySoFactorType.RAW_MAIL);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(HtmlSoFactorType.HTML);

    private final Eml2Html eml2Html;

    public Eml2HtmlExtractor(final IniConfig config) throws ConfigException {
        ImmutableEml2HtmlConfig eml2HtmlConfig =
            new Eml2HtmlConfigBuilder(
                new Eml2HtmlConfigBuilder(config).build())
                .build();
        try {
            eml2Html = new Eml2Html(eml2HtmlConfig);
        } catch (PageHeaderException e) {
            throw new ConfigException(e);
        }
    }

    @Override
    public void close() {
        eml2Html.close();
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        ByteArrayProcessable rawMail =
            inputs.get(0, BinarySoFactorType.RAW_MAIL);
        if (rawMail == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        HtmlNode html;
        try {
            html = eml2Html.convert(rawMail);
        } catch (IOException | MimeException e) {
            callback.failed(e);
            return;
        }
        if (html == null) {
            callback.completed(NULL_RESULT);
        } else {
            callback.completed(
                Collections.singletonList(
                    HtmlSoFactorType.HTML.createFactor(html)));
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        Eml2HtmlExtractorFactory.INSTANCE.registerInternals(registry);
    }
}

