package ru.yandex.mail.so.factors.eml2html;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.annotation.Nonnull;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeTokenStream;
import org.apache.james.mime4j.stream.RecursionMode;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.function.Processable;
import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.io.ByteArrayInputStreamFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.mail.mime.BodyDecoder;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.HtmlSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.sanitizer2.Attr;
import ru.yandex.sanitizer2.AttrUrlInfo;
import ru.yandex.sanitizer2.AttrUrlProcessor;
import ru.yandex.sanitizer2.AttrWithUrls;
import ru.yandex.sanitizer2.BasicAttrs;
import ru.yandex.sanitizer2.BasicStyle;
import ru.yandex.sanitizer2.CssDeclaration;
import ru.yandex.sanitizer2.HtmlCDataTag;
import ru.yandex.sanitizer2.HtmlNode;
import ru.yandex.sanitizer2.HtmlNodeVisitor;
import ru.yandex.sanitizer2.HtmlPrinter;
import ru.yandex.sanitizer2.HtmlTag;
import ru.yandex.sanitizer2.HtmlText;
import ru.yandex.sanitizer2.IdentityAttrPostProcessor;
import ru.yandex.sanitizer2.IdentityCssPostProcessor;
import ru.yandex.sanitizer2.NullUrlCollector;
import ru.yandex.sanitizer2.StringBuilderHtmlCollector;
import ru.yandex.sanitizer2.UrlType;
import ru.yandex.sanitizer2.UrlWithClassInfo;
import ru.yandex.sanitizer2.config.ImmutablePropertyConfig;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.sanitizer2.config.ImmutableUrlConfig;
import ru.yandex.sanitizer2.config.SanitizingConfigBuilder;

public class HtmlImageEmbeddingExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Arrays.asList(
            HtmlSoFactorType.HTML,
            BinarySoFactorType.RAW_MAIL);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(StringSoFactorType.STRING);

    private static final UrlWithClassInfo TRANSPARENT_PNG =
        new UrlWithClassInfo(
            "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAw"
            + "CAAAADklEQVQIHQEDAPz/AAAAAAMAAW7VSZoAAAAASUVORK5CYII=",
            UrlType.CID_URL_TYPE,
            null);

    private final ImmutableSanitizingConfig sanitizingConfig;

    public HtmlImageEmbeddingExtractor(final IniConfig config)
        throws ConfigException
    {
        File sanitizingConfig = config.getInputFile("sanitizing-config");
        try {
            this.sanitizingConfig =
                new SanitizingConfigBuilder(new IniConfig(sanitizingConfig))
                    .build();
        } catch (IOException e) {
            throw new ConfigException(
                "Failed to load sanitizing config from file "
                + sanitizingConfig,
                e);
        }
    }

    public HtmlImageEmbeddingExtractor(
        final ImmutableSanitizingConfig sanitizingConfig)
    {
        this.sanitizingConfig = sanitizingConfig;
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Nonnull
    private Map<String, Attach> extractAttaches(
        @Nonnull final Processable<byte[]> input)
        throws IOException, MimeException
    {
        Map<String, Attach> attaches = null;
        String contentId = null;
        String contentType = null;
        MimeTokenStream tokenStream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        tokenStream.setRecursionMode(RecursionMode.M_NO_RECURSE);
        tokenStream.parse(
            input.processWith(ByteArrayInputStreamFactory.INSTANCE));
        EntityState state = tokenStream.getState();
        while (state != EntityState.T_END_OF_STREAM) {
            switch (state) {
                case T_START_HEADER:
                    contentId = null;
                    contentType = null;
                    break;
                case T_FIELD:
                    Field field = tokenStream.getField();
                    String name = field.getName().toLowerCase(Locale.ROOT);
                    if (name.equals("content-id")) {
                        contentId = field.getBody();
                        int len = contentId.length();
                        if (len > 1
                            && contentId.charAt(0) == '<'
                            && contentId.charAt(len - 1) == '>')
                        {
                            contentId = contentId.substring(1, len - 1);
                        }
                    } else if (name.equals("content-type")) {
                        String value = field.getBody();
                        if (value.startsWith("image/")) {
                            try {
                                contentType =
                                    ContentType.parse(value).getMimeType();
                            } catch (RuntimeException e) {
                                // Ignore invalid Content-Type
                            }
                        }
                    }
                    break;
                case T_BODY:
                    if (contentId != null && contentType != null) {
                        try (InputStream decoded = BodyDecoder.INSTANCE.apply(
                                tokenStream.getInputStream(),
                                tokenStream.getBodyDescriptor()
                                    .getTransferEncoding()))
                        {
                            Attach attach = new Attach(contentType);
                            IOStreamUtils.copy(decoded, attach);
                            if (attaches == null) {
                                attaches = new HashMap<>();
                            }
                            attaches.put(contentId, attach);
                        }
                    }
                    break;
                default:
                    break;
            }
            state = tokenStream.next();
        }
        if (attaches == null) {
            attaches = Collections.emptyMap();
        }
        return attaches;
    }

    public String embedImages(
        @Nonnull final HtmlNode html,
        @Nonnull final Processable<byte[]> input)
        throws IOException, MimeException
    {
        Map<String, Attach> attaches = extractAttaches(input);
        StringBuilderHtmlCollector htmlCollector =
            new StringBuilderHtmlCollector(input.length());
        html
            .accept(new Embedder(sanitizingConfig, attaches))
            .accept(
                new HtmlPrinter<>(
                    sanitizingConfig,
                    htmlCollector,
                    NullUrlCollector.INSTANCE,
                    IdentityAttrPostProcessor.INSTANCE,
                    IdentityCssPostProcessor.INSTANCE));
        return new String(htmlCollector.sb());
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        HtmlNode html = inputs.get(0, HtmlSoFactorType.HTML);
        if (html == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        Processable<byte[]> rawMail =
            inputs.get(1, BinarySoFactorType.RAW_MAIL);
        if (rawMail == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        String fullHtml;
        try {
            fullHtml = embedImages(html, rawMail);
        } catch (IOException | MimeException e) {
            callback.failed(e);
            return;
        }
        callback.completed(
            Collections.singletonList(
                StringSoFactorType.STRING.createFactor(fullHtml)));
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        HtmlImageEmbeddingExtractorFactory.INSTANCE
            .registerInternals(registry);
    }

    private static class Attach extends DecodableByteArrayOutputStream {
        private final String contentType;

        Attach(final String contentType) {
            this.contentType = contentType;
        }
    }

    private static class Embedder
        implements AttrUrlProcessor,
            HtmlNodeVisitor<HtmlNode, RuntimeException>
    {
        private final List<AttrUrlInfo> attrUrls = new ArrayList<>();
        private final Base64Encoder encoder = new Base64Encoder();
        private final ImmutableSanitizingConfig sanitizingConfig;
        private final Map<String, Attach> attaches;
        private StringBuilder sb = null;
        private StringBuilderProcessorAdapter adapter = null;
        private boolean replacedWithTransparentPng;

        Embedder(
            final ImmutableSanitizingConfig sanitizingConfig,
            final Map<String, Attach> attaches)
        {
            this.sanitizingConfig = sanitizingConfig;
            this.attaches = attaches;
        }

        @Override
        public UrlWithClassInfo process(
            final String url,
            final int urlClassId,
            final String urlClassValue)
        {
            if (urlClassId == UrlType.CID_URL_TYPE) {
                // Truncate `cid:' and find attach with such Content-ID
                Attach attach = attaches.get(url.substring(4));
                if (attach == null) {
                    replacedWithTransparentPng = true;
                    return TRANSPARENT_PNG;
                } else {
                    if (sb == null) {
                        sb = new StringBuilder(17 + attach.length() * 4 / 3);
                        adapter = new StringBuilderProcessorAdapter(sb);
                    } else {
                        sb.setLength(0);
                    }
                    sb.append("data:");
                    sb.append(attach.contentType);
                    sb.append(";base64,");
                    attach.processWith(encoder);
                    encoder.processWith(adapter);
                    return new UrlWithClassInfo(
                        sb.toString(),
                        UrlType.CID_URL_TYPE,
                        null);
                }
            } else if (url.startsWith("data:")) {
                return new UrlWithClassInfo(url, urlClassId, urlClassValue);
            } else {
                replacedWithTransparentPng = true;
                return TRANSPARENT_PNG;
            }
        }

        @SuppressWarnings("ReferenceEquality")
        @Override
        public HtmlTag visit(final HtmlTag tag) {
            String tagName = tag.tagName();
            int size = tag.size();
            List<HtmlNode> nodes = new ArrayList<>(size);
            for (int i = 0; i < size; ++i) {
                nodes.add(tag.get(i).accept(this));
            }
            replacedWithTransparentPng = false;
            boolean hasAttrHeight = false;
            boolean hasAttrWidth = false;
            BasicAttrs attrs = new BasicAttrs(tag.attrs());
            for (Map.Entry<String, Attr> entry: attrs.entrySet()) {
                String attrName = entry.getKey();
                Attr attr = entry.getValue();
                if (attrName == "height"
                    && Character.isDigit(attr.value().charAt(0)))
                {
                    hasAttrHeight = true;
                } else if (attrName == "width"
                    && Character.isDigit(attr.value().charAt(0)))
                {
                    hasAttrWidth = true;
                } else {
                    ImmutablePropertyConfig attrConfig = attr.config();
                    ImmutableUrlConfig urlConfig = attrConfig.urlConfig();
                    if (urlConfig != null && urlConfig.image()) {
                        String attrValue = attr.value();
                        attrUrls.clear();
                        attrUrls.add(
                            urlConfig.type().detectUrlClass(
                                0,
                                attrValue.length(),
                                attrValue,
                                sanitizingConfig));
                        entry.setValue(
                            new Attr(
                                new AttrWithUrls(attrValue, attrUrls)
                                    .processWith(this)
                                    .attrValue(),
                                attrConfig));
                    }
                }
            }
            boolean hasCssHeight = false;
            boolean hasCssWidth = false;
            BasicStyle style = new BasicStyle(tag.style());
            for (Map.Entry<String, CssDeclaration> entry: style.entrySet()) {
                String propertyName = entry.getKey();
                CssDeclaration declaration = entry.getValue();
                if (propertyName == "height"
                    && Character.isDigit(declaration.value().charAt(0)))
                {
                    hasCssHeight = true;
                } else if (propertyName == "width"
                    && Character.isDigit(declaration.value().charAt(0)))
                {
                    hasCssWidth = true;
                } else {
                    ImmutablePropertyConfig config = declaration.config();
                    ImmutableUrlConfig urlConfig = config.urlConfig();
                    if (urlConfig != null && urlConfig.image()) {
                        List<AttrUrlInfo> urls = declaration.urls();
                        if (!urls.isEmpty()) {
                            AttrWithUrls attrWithUrls =
                                new AttrWithUrls(declaration.value(), urls)
                                    .processWith(this);
                            entry.setValue(
                                new CssDeclaration(
                                    declaration.property(),
                                    attrWithUrls.attrValue(),
                                    declaration.important(),
                                    config,
                                    attrWithUrls.urls()));
                        }
                    }
                }
            }
            if (replacedWithTransparentPng) {
                if ((hasAttrHeight || hasCssHeight)
                    && !(hasAttrWidth || hasCssWidth))
                {
                    if (hasAttrHeight) {
                        attrs.put(
                            "width",
                            new Attr(
                                "0",
                                sanitizingConfig.tags().get(tagName).attrs()
                                    .get("width")));
                        style.remove("width");
                    } else {
                        style.put(
                            "width",
                            new CssDeclaration(
                                "width",
                                "0px",
                                false,
                                sanitizingConfig.style().get("width"),
                                Collections.emptyList()));
                        attrs.remove("width");
                    }
                } else if ((hasAttrWidth || hasCssWidth)
                    && !(hasAttrHeight || hasCssHeight))
                {
                    if (hasAttrWidth) {
                        attrs.put(
                            "height",
                            new Attr(
                                "0",
                                sanitizingConfig.tags().get(tagName).attrs()
                                    .get("height")));
                        style.remove("height");
                    } else {
                        style.put(
                            "height",
                            new CssDeclaration(
                                "height",
                                "0px",
                                false,
                                sanitizingConfig.style().get("height"),
                                Collections.emptyList()));
                        attrs.remove("height");
                    }
                }
            }
            return new HtmlTag(
                nodes,
                tag.escaping(),
                tagName,
                tag.classes(),
                tag.sanitizedClasses(),
                tag.id(),
                tag.sanitizedId(),
                attrs,
                style);
        }

        @Override
        public HtmlCDataTag visit(final HtmlCDataTag cdata) {
            return cdata;
        }

        @Override
        public HtmlText visit(final HtmlText text) {
            return text;
        }
    }
}

