package ru.yandex.mail.so.factors.extractors;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.mail.so.factors.BasicSoFunctionInputs;
import ru.yandex.mail.so.factors.ConcatSoFunctionInputs;
import ru.yandex.mail.so.factors.IdentitySoFactorFieldAccessor;
import ru.yandex.mail.so.factors.Remapping;
import ru.yandex.mail.so.factors.RemappingSoFunctionInputs;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFactorFieldAccessor;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.dsl.SoCallArgument;
import ru.yandex.mail.so.factors.dsl.SoCallDeclaration;
import ru.yandex.mail.so.factors.dsl.SoConst;
import ru.yandex.mail.so.factors.dsl.SoConsts;
import ru.yandex.mail.so.factors.dsl.SoReturnDeclaration;
import ru.yandex.mail.so.factors.dsl.SoVariables;
import ru.yandex.mail.so.factors.dsl.ValueInfo;
import ru.yandex.mail.so.factors.predicates.NotPredicate;
import ru.yandex.mail.so.factors.predicates.SimpleSoPredicateType;
import ru.yandex.mail.so.factors.predicates.SoPredicate;
import ru.yandex.mail.so.factors.types.NullSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.xpath.XPathParser;
import ru.yandex.util.timesource.TimeSource;

public class ChainExtractor implements SoFactorsExtractor {
    private final List<SoFactorType<?>> inputs;
    private final int inputsSize;
    private final List<SoFactorType<?>> outputs;
    private final List<SoFactor<?>> consts;
    private final SoFactorsExtractorsRegistry registry;
    // List output variables and their predicates
    private final OutputPredicate[] outputPredicates;
    private final int[] outputVariables;
    private final SoFactorFieldAccessor[] outputFieldAccessors;
    private final Node[] variableIdToNode;
    private final int outputVariablesCount;
    private final String[] variableIdToName;
    private final int startNodesCount;

    public ChainExtractor(
        final String name,
        final List<SoFunctionArgumentInfo> inputs,
        final List<SoFactorType<?>> outputs,
        final List<SoCallDeclaration> calls,
        final List<SoReturnDeclaration> returns,
        final SoFactorsExtractorFactoryContext context)
        throws ConfigException
    {
        this.outputs = outputs;
        SoConsts consts = new SoConsts();
        for (SoCallDeclaration callDeclaration: calls) {
            for (SoCallArgument callArgument: callDeclaration.arguments()) {
                consts.addConst(callArgument);
            }
        }
        for (SoReturnDeclaration returnDeclaration: returns) {
            for (SoCallArgument callArgument: returnDeclaration.returns()) {
                consts.addConst(callArgument);
            }
            for (SoCallArgument callArgument: returnDeclaration.elses()) {
                consts.addConst(callArgument);
            }
            for (SoCallArgument callArgument
                : returnDeclaration.predicateArguments())
            {
                consts.addConst(callArgument);
            }
        }
        this.consts = consts.constsList();
        int inputsSize = inputs.size();
        this.inputs = new ArrayList<>(inputsSize);
        SoVariables inputVariables = new SoVariables(consts);
        for (int i = 0; i < inputsSize; ++i) {
            SoFunctionArgumentInfo input = inputs.get(i);
            String inputName = input.name();
            SoFactorType<?> type = input.type();
            inputVariables.addVariable(inputName, type, "Input");
            this.inputs.add(type);
        }
        this.inputsSize = inputsSize + consts.size();
        registry = context.registry();
        SoVariables variables = new SoVariables(inputVariables);
        List<OutputPredicate> outputPredicates =
            new ArrayList<>(outputs.size());

        // Map variable id to node which will calculate this variable
        Map<Integer, Node> variableDependencies = new HashMap<>();
        int startNodesCount = 0;
        Function<String, Consumer<ExtractorStat>> statsConsumerFactory =
            context.statsConsumerFactory();
        for (SoCallDeclaration callDeclaration: calls) {
                Call call = new Call(callDeclaration, registry, variables);
                Consumer<ExtractorStat> statsConsumer;
                if (call.trace) {
                    statsConsumer = statsConsumerFactory.apply(call.name);
                } else {
                    statsConsumer = null;
                }
                Set<Node> dependeeNodes =
                    new HashSet<>(call.inputs.size() << 1);
                for (int input: call.inputsRemapping.indexesRemapping()) {
                    Node dependeeNode = variableDependencies.get(input);
                    if (dependeeNode != null) {
                        dependeeNodes.add(dependeeNode);
                    }
                }
                int inputsCount = 0;
                int singleInput = -1;
                for (int input: call.inputsRemapping.indexesRemapping()) {
                    if (input >= this.inputsSize) {
                        singleInput = input;
                        ++inputsCount;
                    }
                }
                if (inputsCount == 0) {
                    ++startNodesCount;
                }
                Node node = new Node(
                    call.name,
                    dependeeNodes.toArray(new Node[dependeeNodes.size()]),
                    call.extractor,
                    call.inputsRemapping,
                    singleInput,
                    inputsCount,
                    call.outputsRemapping,
                    call.async,
                    call.critical,
                    statsConsumer);
                for (int output: call.outputsRemapping) {
                    variableDependencies.put(output, node);
                }
        }
        int outputVariablesCount = 0;
        int predicatesCount = 0;
        for (SoReturnDeclaration returnDeclaration: returns) {
            Return functionReturn =
                new Return(
                    returnDeclaration,
                    registry,
                    inputVariables,
                    variables,
                    consts);
            List<SoCallArgument> returnArguments = returnDeclaration.returns();
            int returnsCount = returnArguments.size();
            if (outputVariablesCount + returnsCount > outputs.size()) {
                throw new ConfigException(
                    "Unexpected returns "
                    + returnArguments + ": "
                    + outputVariablesCount + " + " + returnsCount
                    + " > " + outputs.size());
            }
            for (int i = 0; i < returnsCount; ++i) {
                SoFactorType<?> expectedType =
                    outputs.get(outputVariablesCount++);
                SoFactorType<?> returnType = functionReturn.returnTypes.get(i);
                if (returnType != NullSoFactorType.NULL
                    && returnType != expectedType)
                {
                    throw new ConfigException(
                        "Return " + returnArguments.get(i)
                        + " type mismatch. Expected " + expectedType
                        + ", found: " + returnType);
                }
            }
            if (functionReturn.predicate != null) {
                ++predicatesCount;
            }
            outputPredicates.add(
                new OutputPredicate(
                    functionReturn.returnIds,
                    functionReturn.returnFieldAccessors,
                    functionReturn.elseIds,
                    functionReturn.elseFieldAccessors,
                    functionReturn.predicate,
                    functionReturn.predicateInputsRemapping));
        }
        int size = outputs.size();
        if (outputVariablesCount != size) {
            throw new ConfigException(
                "Not enough returns. Required " + size
                + ", declared " + outputVariablesCount);
        }
        this.outputVariablesCount = outputVariablesCount;
        variableIdToNode = new Node[variables.size()];
        if (predicatesCount == 0) {
            this.outputPredicates = null;
            outputVariables = new int[size];
            outputFieldAccessors = new SoFactorFieldAccessor[size];
            int pos = 0;
            for (OutputPredicate outputPredicate: outputPredicates) {
                for (int i = 0; i < outputPredicate.variableIds.length; ++i) {
                    outputVariables[pos] = outputPredicate.variableIds[i];
                    outputFieldAccessors[pos] =
                        outputPredicate.fieldAccessors[i];
                    ++pos;
                }
            }
            variableIdToName = null;
        } else {
            this.outputPredicates =
                outputPredicates.toArray(new OutputPredicate[predicatesCount]);
            outputVariables = null;
            outputFieldAccessors = null;
            variableIdToName = variables.variableIdToName();
        }
        for (Map.Entry<Integer, Node> entry: variableDependencies.entrySet()) {
            variableIdToNode[entry.getKey().intValue()] = entry.getValue();
        }
        this.startNodesCount = startNodesCount;
    }

    private void populateRequiredNodes(
        final ChainExecutionContext chainContext,
        final IdentityHashMap<Node, Node> requiredNodes,
        final List<Node> startNodes,
        final List<List<FutureCallback<Object>>> callbacks,
        final List<MultiFutureCallback<Object>> undoneCallbacks,
        final Node node)
    {
        if (requiredNodes.put(node, node) == null) {
            switch (node.inputsCount) {
                case 0:
                    startNodes.add(node);
                    break;
                case 1:
                    callbacks.get(node.singleInput).add(
                        new BarrierCallback(
                            new NodeCallback(node, chainContext)));
                    break;
                default:
                    MultiFutureCallback<Object> nodeCallback =
                        new MultiFutureCallback<>(
                            new BarrierCallback(
                                new NodeCallback(node, chainContext)));
                    undoneCallbacks.add(nodeCallback);
                    for (int input: node.inputsRemapping.indexesRemapping()) {
                        // Do not create callbacks for input variables
                        if (input >= inputsSize) {
                            callbacks.get(input).add(
                                nodeCallback.newCallback());
                        }
                    }
                    break;
            }
            for (Node dependeeNode: node.dependeeNodes) {
                populateRequiredNodes(
                    chainContext,
                    requiredNodes,
                    startNodes,
                    callbacks,
                    undoneCallbacks,
                    dependeeNode);
            }
        }
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext extractorContext,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        SoFunctionInputs constsAndInputs;
        if (consts.isEmpty()) {
            constsAndInputs = inputs;
        } else {
            constsAndInputs =
                new ConcatSoFunctionInputs(
                    new BasicSoFunctionInputs(
                        extractorContext.accessViolationHandler(),
                        consts),
                    inputs);
        }
        int[] outputVariables;
        SoFactorFieldAccessor[] outputFieldAccessors;
        if (variableIdToName == null) {
            outputVariables = this.outputVariables;
            outputFieldAccessors = this.outputFieldAccessors;
        } else {
            outputVariables = new int[outputVariablesCount];
            outputFieldAccessors =
                new SoFactorFieldAccessor[outputVariablesCount];
            StringBuilder sb =
                new StringBuilder("Returns that will be calculated:");
            int pos = 0;
            for (OutputPredicate outputPredicate: outputPredicates) {
                int[] outputIds;
                SoFactorFieldAccessor[] fieldAccessors;
                if (outputPredicate.predicate == null) {
                    outputIds = outputPredicate.variableIds;
                    fieldAccessors = outputPredicate.fieldAccessors;
                } else {
                    SoFunctionInputs predicateInputs =
                        new RemappingSoFunctionInputs(
                            extractorContext.accessViolationHandler(),
                            constsAndInputs,
                            outputPredicate.predicateInputsRemapping);
                    boolean predicatePassed = outputPredicate.predicate.test(
                        extractorContext,
                        predicateInputs);
                    if (predicatePassed) {
                        outputIds = outputPredicate.variableIds;
                        fieldAccessors = outputPredicate.fieldAccessors;
                    } else {
                        outputIds = outputPredicate.elseIds;
                        fieldAccessors = outputPredicate.elseAccessors;
                    }
                }
                for (int i = 0; i < outputIds.length; ++i) {
                    int outputId = outputIds[i];
                    outputVariables[pos] = outputId;
                    SoFactorFieldAccessor fieldAccessor = fieldAccessors[i];
                    outputFieldAccessors[pos] = fieldAccessor;
                    ++pos;
                    if (outputId == -1) {
                        sb.append(" null");
                    } else {
                        sb.append(' ');
                        sb.append(variableIdToName[outputId]);
                        if (fieldAccessor != null) {
                            fieldAccessor.toStringBuilder(sb);
                        }
                    }
                }
            }
            extractorContext.logger().info(new String(sb));
        }
        // For each variable (local and input) this list will contains list of
        // callbacks awaiting for this variable to be computed
        List<List<FutureCallback<Object>>> callbacks =
            new ArrayList<>(variableIdToNode.length);
        for (int i = 0; i < variableIdToNode.length; ++i) {
            callbacks.add(new ArrayList<>(4));
        }
        FutureCallback<? super List<SoFactor<?>>> idempotentCallback =
            new IdempotentFutureCallback<>(callback);
        ChainExecutionContext chainContext = new ChainExecutionContext(
            extractorContext,
            idempotentCallback,
            callbacks,
            constsAndInputs);
        MultiFutureCallback<Object> finalCallback =
            new MultiFutureCallback<>(
                new FinalCallback(
                    idempotentCallback,
                    chainContext.allInputs,
                    outputVariables,
                    outputFieldAccessors));
        for (int outputVariable: outputVariables) {
            // Do not create callback for -1 variables and for input variables
            if (outputVariable >= inputsSize) {
                callbacks.get(outputVariable).add(finalCallback.newCallback());
            }
        }
        IdentityHashMap<Node, Node> requiredNodesMap =
            new IdentityHashMap<>(variableIdToNode.length << 1);
        List<Node> startNodes = new ArrayList<>(startNodesCount);
        List<MultiFutureCallback<Object>> undoneCallbacks =
            new ArrayList<>(variableIdToNode.length);
        for (int outputVariable: outputVariables) {
            if (outputVariable >= inputsSize) {
                populateRequiredNodes(
                    chainContext,
                    requiredNodesMap,
                    startNodes,
                    callbacks,
                    undoneCallbacks,
                    variableIdToNode[outputVariable]);
            }
        }
        int startNodesSize = startNodes.size();
        for (int i = 0; i < startNodesSize; ++i) {
            new NodeCallback(startNodes.get(i), chainContext).start();
        }
        int undoneCallbacksSize = undoneCallbacks.size();
        for (int i = 0; i < undoneCallbacksSize; ++i) {
            undoneCallbacks.get(i).done();
        }
        finalCallback.done();
    }

    @Override
    public void close() throws IOException {
        registry.close();
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return outputs;
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry) {
    }

    private static class OutputPredicate {
        private final int[] variableIds;
        private final SoFactorFieldAccessor[] fieldAccessors;
        private final int[] elseIds;
        private final SoFactorFieldAccessor[] elseAccessors;
        private final SoPredicate predicate;
        private final Remapping predicateInputsRemapping;

        OutputPredicate(
            final int[] variableIds,
            final SoFactorFieldAccessor[] fieldAccessors,
            final int[] elseIds,
            final SoFactorFieldAccessor[] elseAccessors,
            final SoPredicate predicate,
            final Remapping predicateInputsRemapping)
        {
            this.variableIds = variableIds;
            this.fieldAccessors = fieldAccessors;
            this.elseIds = elseIds;
            this.elseAccessors = elseAccessors;
            this.predicate = predicate;
            this.predicateInputsRemapping = predicateInputsRemapping;
        }
    }

    private static class Node {
        private final String name;
        private final Node[] dependeeNodes;
        private final SoFactorsExtractor extractor;
        private final Remapping inputsRemapping;
        private final int singleInput;
        private final int inputsCount;
        private final int[] outputsRemapping;
        private final boolean async;
        private final boolean critical;
        private final Consumer<ExtractorStat> statsConsumer;

        Node(
            final String name,
            final Node[] dependeeNodes,
            final SoFactorsExtractor extractor,
            final Remapping inputsRemapping,
            final int singleInput,
            final int inputsCount,
            final int[] outputsRemapping,
            final boolean async,
            final boolean critical,
            final Consumer<ExtractorStat> statsConsumer)
        {
            this.name = name;
            this.dependeeNodes = dependeeNodes;
            this.extractor = extractor;
            this.inputsRemapping = inputsRemapping;
            this.singleInput = singleInput;
            this.inputsCount = inputsCount;
            this.outputsRemapping = outputsRemapping;
            this.async = async;
            this.critical = critical;
            this.statsConsumer = statsConsumer;
        }
    }

    private static class ChainExecutionContext {
        private final SoFactorsExtractorContext extractorContext;
        private final FutureCallback<? super List<SoFactor<?>>> chainCallback;
        private final List<List<FutureCallback<Object>>> callbacks;
        private final int localVariablesOffset;
        private final BasicSoFunctionInputs localVariables;
        private final SoFunctionInputs allInputs;

        ChainExecutionContext(
            final SoFactorsExtractorContext extractorContext,
            final FutureCallback<? super List<SoFactor<?>>> chainCallback,
            final List<List<FutureCallback<Object>>> callbacks,
            final SoFunctionInputs inputs)
        {
            this.extractorContext = extractorContext;
            this.chainCallback = chainCallback;
            this.callbacks = callbacks;
            localVariablesOffset = inputs.size();
            localVariables =
                new BasicSoFunctionInputs(
                    extractorContext.accessViolationHandler(),
                    callbacks.size() - localVariablesOffset);
            allInputs = new ConcatSoFunctionInputs(inputs, localVariables);
        }
    }

    private static class FinalCallback
        extends AbstractFilterFutureCallback<Object, List<SoFactor<?>>>
    {
        private final SoFunctionInputs inputs;
        private final int[] outputVariables;
        private final SoFactorFieldAccessor[] outputFieldAccessors;

        FinalCallback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final SoFunctionInputs inputs,
            final int[] outputVariables,
            final SoFactorFieldAccessor[] outputFieldAccessors)
        {
            super(callback);
            this.inputs = inputs;
            this.outputVariables = outputVariables;
            this.outputFieldAccessors = outputFieldAccessors;
        }

        @Override
        public void completed(final Object result) {
            List<SoFactor<?>> output = new ArrayList<>(outputVariables.length);
            for (int i = 0; i < outputVariables.length; ++i) {
                int outputVariable = outputVariables[i];
                if (outputVariable >= 0) {
                    SoFactor<?> factor = inputs.get(outputVariable);
                    if (factor == null) {
                        output.add(null);
                    } else {
                        SoFactorFieldAccessor fieldAccessor =
                            outputFieldAccessors[i];
                        if (fieldAccessor == null) {
                            output.add(factor);
                        } else {
                            Object value =
                                fieldAccessor.extractField(
                                    factor.value(),
                                    inputs.accessViolationHandler());
                            if (value == null) {
                                output.add(null);
                            } else {
                                SoFactor<?> outputFactor =
                                    fieldAccessor.fieldType().tryCreateFactor(
                                        value);
                                if (outputFactor == null) {
                                    inputs
                                        .accessViolationHandler()
                                        .handleBadFieldAccessorFieldType(
                                            i,
                                            factor,
                                            fieldAccessor,
                                            value);
                                }
                                output.add(outputFactor);
                            }
                        }
                    }
                } else {
                    output.add(null);
                }
            }
            callback.completed(output);
        }
    }

    private static class BarrierCallback implements FutureCallback<Object> {
        private final NodeCallback nodeCallback;

        BarrierCallback(final NodeCallback nodeCallback) {
            this.nodeCallback = nodeCallback;
        }

        @Override
        public void cancelled() {
            nodeCallback.cancelled();
        }

        @Override
        public void completed(final Object result) {
            try {
                nodeCallback.start();
            } catch (RuntimeException e) {
                nodeCallback.failed(e);
            }
        }

        @Override
        public void failed(final Exception e) {
            nodeCallback.failed(e);
        }
    }

    private static class NodeCallback
        implements FutureCallback<List<SoFactor<?>>>, Runnable
    {
        private final Node node;
        private final ChainExecutionContext chainContext;
        private volatile long start;

        NodeCallback(
            final Node node,
            final ChainExecutionContext chainContext)
        {
            this.node = node;
            this.chainContext = chainContext;
        }

        public void start() {
            if (chainContext.extractorContext.cancelled()) {
                cancelled();
            } else if (node.async) {
                chainContext.extractorContext.executor().execute(this);
            } else {
                if (chainContext.extractorContext.debugExtractors()) {
                    chainContext.extractorContext.logger().fine(
                        "Starting node " + node.name);
                }
                start = TimeSource.INSTANCE.currentTimeMillis();
                try {
                    node.extractor.extract(
                        new PrefixedSoFactorsExtractorContext(
                            chainContext.extractorContext,
                            node.name),
                        new RemappingSoFunctionInputs(
                            chainContext.extractorContext
                                .accessViolationHandler(),
                            chainContext.allInputs,
                            node.inputsRemapping),
                        this);
                } catch (Exception e) {
                    failed(e);
                } catch (Error e) {
                    failed(new Exception(e));
                }
            }
        }

        private long timeTaken() {
            long start = this.start;
            if (start == 0L) {
                return 0L;
            } else {
                return TimeSource.INSTANCE.currentTimeMillis() - start;
            }
        }

        private void done() {
            for (int output: node.outputsRemapping) {
                List<FutureCallback<Object>> callbacks =
                    chainContext.callbacks.get(output);
                int size = callbacks.size();
                for (int i = 0; i < size; ++i) {
                    callbacks.get(i).completed(null);
                }
            }
        }

        @Override
        public void run() {
            if (chainContext.extractorContext.debugExtractors()) {
                chainContext.extractorContext.logger().fine(
                    "Starting node " + node.name);
            }
            start = TimeSource.INSTANCE.currentTimeMillis();
            try {
                node.extractor.extract(
                    chainContext.extractorContext,
                    new RemappingSoFunctionInputs(
                        chainContext.extractorContext.accessViolationHandler(),
                        chainContext.allInputs,
                        node.inputsRemapping),
                    this);
            } catch (Exception e) {
                failed(e);
            } catch (Error e) {
                failed(new Exception(e));
            }
        }

        @Override
        public void cancelled() {
            long timeTaken = timeTaken();
            chainContext.extractorContext.logger().warning(
                "Node " + node.name
                + " execution cancelled after " + timeTaken + " ms");
            if (node.statsConsumer != null) {
                node.statsConsumer.accept(
                    new ExtractorStat(timeTaken, 0L, 0L, 0L, 1L));
            }
            if (node.critical) {
                chainContext.chainCallback.failed(new CancellationException());
            } else {
                done();
            }
        }

        @Override
        public void completed(final List<SoFactor<?>> results) {
            int size = results.size();
            if (size != node.outputsRemapping.length) {
                chainContext
                    .extractorContext
                    .accessViolationHandler()
                    .violationsCounter()
                    .increment();
                failed(
                    new IllegalArgumentException(
                        "Expected " + node.outputsRemapping.length
                        + " results from " + node.extractor
                        + ", got " + size + ':' + ' ' + results
                        + ". From here"));
            } else {
                int setCount = chainContext.localVariables.set(
                    results,
                    node.outputsRemapping,
                    chainContext.localVariablesOffset);
                if (node.statsConsumer != null
                    || chainContext.extractorContext.debugExtractors())
                {
                    long timeTaken = timeTaken();
                    if (chainContext.extractorContext.logger()
                        .isLoggable(Level.INFO))
                    {
                        chainContext.extractorContext.logger().info(
                            "Node " + node.name + " completed execution in "
                            + timeTaken + " ms. "
                            + setCount + " factors(s) extracted");
                    }
                    if (node.statsConsumer != null) {
                        node.statsConsumer.accept(
                            new ExtractorStat(
                                timeTaken,
                                setCount,
                                1L,
                                0L,
                                0L));
                    }
                }
                done();
            }
        }

        @Override
        public void failed(final Exception e) {
            long timeTaken = timeTaken();
            chainContext.extractorContext.logger().log(
                Level.WARNING,
                "Node " + node.name
                + " execution failed after " + timeTaken + " ms",
                e);
            if (node.statsConsumer != null) {
                node.statsConsumer.accept(
                    new ExtractorStat(timeTaken, 0L, 0L, 1L, 0L));
            }
            if (node.critical) {
                chainContext.chainCallback.failed(e);
            } else {
                done();
            }
        }
    }

    private static class Call {
        private final String name;
        private final SoFactorsExtractor extractor;
        private final List<SoCallArgument> inputs;
        private final Remapping inputsRemapping;
        private final List<SoCallArgument> outputs;
        private final int[] outputsRemapping;
        private final boolean async;
        private final boolean trace;
        private final boolean critical;

        Call(
            final SoCallDeclaration declaration,
            final SoFactorsExtractorsRegistry registry,
            final SoVariables variables)
            throws ConfigException
        {
            String extractorName = declaration.functionName();
            String name = declaration.alias();
            if (name == null) {
                this.name = extractorName;
            } else {
                this.name = name;
            }
            inputs = declaration.arguments();
            String description = "Function <" + extractorName + '>';
            SoFactorsExtractor extractor =
                registry.getExtractor(extractorName);
            if (extractor == null) {
                SimpleSoFactorsExtractorFactory simpleFactory =
                    registry.getSimpleExtractorFactory(extractorName);
                if (simpleFactory == null) {
                    throw new ConfigException(
                        "Unknown extractor <" + extractorName + '>');
                } else {
                    int size = inputs.size();
                    List<SoFactorType<?>> types = new ArrayList<>(size);
                    for (int i = 0; i < size; ++i) {
                        types.add(
                            variables.getArgumentType(
                                inputs.get(i),
                                description));
                    }
                    extractor = simpleFactory.createExtractor(types);
                    registry.addAnonymousExtractor(extractor);
                }
            } else {
                List<SoFactorType<?>> extractorInputs = extractor.inputs();
                variables.checkArguments(inputs, extractorInputs, description);
            }
            this.extractor = extractor;
            inputsRemapping =
                variables.createRemapping(inputs, description + " input");
            List<String> outputNames = declaration.outputs();
            int outputsSize = outputNames.size();
            outputs = new ArrayList<>(outputsSize);
            for (int i = 0; i < outputsSize; ++i) {
                outputs.add(
                    new SoCallArgument(
                        outputNames.get(i),
                        SoCallArgument.Type.TOKEN));
            }
            List<SoFactorType<?>> outputTypes = extractor.outputs();
            int size = outputTypes.size();
            if (size != outputsSize) {
                throw new ConfigException(
                    "Extractor <" + extractorName
                    + "> has invalid number of outputs, expected "
                    + size + " outputs with types " + outputTypes
                    + ", found " + outputsSize + " with names "
                    + outputs);
            }
            for (int i = 0; i < size; ++i) {
                SoFactorType<?> outputType = outputTypes.get(i);
                variables.addVariable(
                    outputNames.get(i),
                    outputType,
                    description);
            }
            outputsRemapping =
                variables.createRemapping(
                    outputs,
                    description + " output")
                    .indexesRemapping();
            async = declaration.async();
            trace = declaration.trace();
            critical = declaration.critical();
        }
    }

    private static class Return {
        private final int[] returnIds;
        private final SoFactorFieldAccessor[] returnFieldAccessors;
        private final List<SoFactorType<?>> returnTypes;
        private final int[] elseIds;
        private final SoFactorFieldAccessor[] elseFieldAccessors;
        private final SoPredicate predicate;
        private final Remapping predicateInputsRemapping;

        public Return(
            final SoReturnDeclaration declaration,
            final SoFactorsExtractorsRegistry registry,
            final SoVariables inputs,
            // It is expected that variables includes all input variables
            final SoVariables variables,
            final SoConsts consts)
            throws ConfigException
        {
            List<SoCallArgument> returns = declaration.returns();
            List<SoCallArgument> predicateInputs =
                declaration.predicateArguments();
            String predicateName = declaration.predicateName();
            String errorDescription =
                "Predicate <" + predicateName + "> input";
            SoPredicate predicate;
            if (predicateName == null) {
                predicate = null;
            } else {
                predicate = registry.getPredicate(predicateName);
                if (predicate == null) {
                    SimpleSoPredicateType predicateType =
                        registry.getSimplePredicateType(predicateName);
                    if (predicateType == null) {
                        throw new ConfigException(
                            "Unknown predicate <" + predicateName + '>');
                    } else {
                        int size = predicateInputs.size();
                        List<SoFactorType<?>> types = new ArrayList<>(size);
                        for (int i = 0; i < size; ++i) {
                            types.add(
                                variables.getArgumentType(
                                    predicateInputs.get(i),
                                    errorDescription));
                        }
                        predicate = predicateType.create(types);
                        registry.addAnonymousPredicate(predicate);
                    }
                }
                if (declaration.invertPredicate()) {
                    predicate = new NotPredicate(predicate);
                }
            }
            this.predicate = predicate;
            int size = returns.size();
            returnIds = new int[size];
            returnFieldAccessors = new SoFactorFieldAccessor[size];
            returnTypes = new ArrayList<>(size);
            elseIds = new int[size];
            elseFieldAccessors = new SoFactorFieldAccessor[size];
            for (int i = 0; i < size; ++i) {
                SoCallArgument callArgument = returns.get(i);
                if (callArgument.type() == SoCallArgument.Type.TOKEN) {
                    ValueInfo info = new ValueInfo(variables, "Return");
                    XPathParser.parse('.' + callArgument.image(), info);
                    returnIds[i] = info.variableInfo().id();
                    SoFactorFieldAccessor fieldAccessor = info.fieldAccessor();
                    if (fieldAccessor instanceof IdentitySoFactorFieldAccessor) {
                        returnTypes.add(info.variableInfo().type());
                    } else {
                        returnFieldAccessors[i] = fieldAccessor;
                        returnTypes.add(fieldAccessor.fieldType());
                    }
                } else {
                    SoConst constValue = consts.getConst(callArgument);
                    returnIds[i] = constValue.id();
                    returnTypes.add(constValue.value().type());
                }
            }
            if (predicate == null) {
                predicateInputsRemapping = null;
                Arrays.fill(elseIds, -1);
            } else {
                List<SoCallArgument> elses = declaration.elses();
                if (elses.isEmpty()) {
                    Arrays.fill(elseIds, -1);
                } else {
                    // returns and elses sizes were already checked in parser
                    for (int i = 0; i < size; ++i) {
                        SoCallArgument callArgument = elses.get(i);
                        SoFactorType<?> returnType = returnTypes.get(i);
                        SoFactorType<?> elseType;
                        if (callArgument.type() == SoCallArgument.Type.TOKEN) {
                            ValueInfo info = new ValueInfo(variables, "Else");
                            XPathParser.parse('.' + callArgument.image(), info);
                            elseIds[i] = info.variableInfo().id();
                            SoFactorFieldAccessor fieldAccessor =
                                info.fieldAccessor();
                            if (fieldAccessor instanceof IdentitySoFactorFieldAccessor) {
                                elseType = info.variableInfo().type();
                            } else {
                                elseFieldAccessors[i] = fieldAccessor;
                                elseType = fieldAccessor.fieldType();
                            }
                        } else {
                            SoConst constValue = consts.getConst(callArgument);
                            elseType = constValue.value().type();
                            elseIds[i] = constValue.id();
                        }
                        if (returnType == NullSoFactorType.NULL) {
                            returnTypes.set(i, elseType);
                        } else if (elseType != NullSoFactorType.NULL
                            && elseType != returnType)
                        {
                            throw new ConfigException(
                                "Else variable " + callArgument
                                + " type mismatch with variable "
                                + returns.get(i)
                                + ", expected " + returnType
                                + ", found " + elseType);
                        }
                    }
                }
                inputs.checkArguments(
                    predicateInputs,
                    predicate.inputs(),
                    errorDescription);
                predicateInputsRemapping = variables.createRemapping(
                    predicateInputs,
                    errorDescription);
            }
        }
    }
}

