package ru.yandex.mail.so.factors.extractors;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.BooleanSoFactorType;
import ru.yandex.mail.so.factors.types.JsonMapSoFactorType;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.LongSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocSoFactorType;
import ru.yandex.parser.config.ConfigException;

public class ComposeDocExtractor implements SoFactorsExtractor {
    private final int size;
    private final List<SoFactorType<?>> inputs;
    private final List<SoFactorType<?>> outputs;
    private final String[] fieldNames;
    private final boolean tikaiteDocInput;
    private final boolean tikaiteDocOutput;

    public ComposeDocExtractor(
        final List<SoFunctionArgumentInfo> inputs,
        final List<SoFactorType<?>> outputs)
        throws ConfigException
    {
        size = inputs.size();
        this.inputs = new ArrayList<>(size);
        fieldNames = new String[size];
        if (inputs.size() < 1) {
            throw new ConfigException(
                "At least one input expected, got: " + inputs);
        }
        tikaiteDocInput =
            inputs.get(0).type() == TikaiteDocSoFactorType.TIKAITE_DOC;
        if (tikaiteDocInput) {
            this.inputs.add(TikaiteDocSoFactorType.TIKAITE_DOC);
        } else {
            SoFactorsExtractor.forceInputType(
                JsonMapSoFactorType.JSON_MAP,
                inputs,
                0);
            this.inputs.add(JsonMapSoFactorType.JSON_MAP);
        }
        for (int i = 1; i < size; ++i) {
            SoFunctionArgumentInfo input = inputs.get(i);
            SoFactorType<?> type = input.type();
            if (type != StringSoFactorType.STRING
                && type != LongSoFactorType.LONG
                && type != BooleanSoFactorType.BOOLEAN)
            {
                SoFactorsExtractor.forceInputType(
                    JsonObjectSoFactorType.JSON_OBJECT,
                    inputs,
                    i);
            }
            fieldNames[i] = input.name();
            this.inputs.add(type);
        }
        int outputsSize = outputs.size();
        if (outputsSize != 1) {
            throw new ConfigException(
                "Exactly one output expected, got: " + outputs);
        }
        tikaiteDocOutput =
            outputs.get(0) == TikaiteDocSoFactorType.TIKAITE_DOC;
        if (tikaiteDocOutput) {
            this.outputs =
                Collections.singletonList(TikaiteDocSoFactorType.TIKAITE_DOC);
        } else {
            this.outputs =
                Collections.singletonList(JsonMapSoFactorType.JSON_MAP);
        }
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return outputs;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        JsonMap doc;
        if (tikaiteDocInput) {
            doc = inputs.get(0, TikaiteDocSoFactorType.TIKAITE_DOC);
        } else {
            doc = inputs.get(0, JsonMapSoFactorType.JSON_MAP);
        }
        if (doc == null) {
            context.logger().warning("No tikaite doc found in input");
            callback.completed(NULL_RESULT);
        } else {
            JsonMap copy = doc.deepCopy();
            for (int i = 1; i < size; ++i) {
                SoFactorType<?> type = this.inputs.get(i);
                JsonObject jsonValue = null;
                if (type == StringSoFactorType.STRING) {
                    String value = inputs.get(i, StringSoFactorType.STRING);
                    if (value != null) {
                        jsonValue = new JsonString(value);
                    }
                } else if (type == LongSoFactorType.LONG) {
                    Long value = inputs.get(i, LongSoFactorType.LONG);
                    if (value != null) {
                        jsonValue = new JsonLong(value.longValue());
                    }
                } else if (type == BooleanSoFactorType.BOOLEAN) {
                    Boolean value = inputs.get(i, BooleanSoFactorType.BOOLEAN);
                    if (value != null) {
                        jsonValue = JsonBoolean.valueOf(value.booleanValue());
                    }
                } else {
                    jsonValue =
                        inputs.get(i, JsonObjectSoFactorType.JSON_OBJECT);
                }
                if (jsonValue != null) {
                    copy.put(fieldNames[i], jsonValue);
                }
            }
            SoFactor<?> result;
            if (tikaiteDocOutput) {
                result = TikaiteDocSoFactorType.TIKAITE_DOC.createFactor(copy);
            } else {
                result = JsonMapSoFactorType.JSON_MAP.createFactor(copy);
            }
            callback.completed(Collections.singletonList(result));
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        ComposeDocExtractorFactory.INSTANCE.registerInternals(registry);
    }
}

