package ru.yandex.mail.so.factors.extractors;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public enum ConcatExtractorFactory
    implements IniConfigSoFactorsExtractorFactory
{
    INSTANCE;

    @Override
    public void close() {
    }

    @Override
    public SoFactorsExtractor createIniConfigExtractor(
        final String name,
        final List<SoFunctionArgumentInfo> inputs,
        final List<SoFactorType<?>> outputs,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        String delimiter = config.getString("delimiter", "");
        int size = inputs.size();
        switch (delimiter.length()) {
            case 0:
                return new BareConcatExtractor(size);
            case 1:
                char c = delimiter.charAt(0);
                return new CharConcatExtractor(size, c);
            default:
                return new ConcatExtractor(size, delimiter);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        registry.typesRegistry().registerSoFactorType(
            StringSoFactorType.STRING);
    }

    private static abstract class ConcatExtractorBase
        implements SoFactorsExtractor
    {
        private static final List<SoFactorType<?>> OUTPUTS =
            Collections.singletonList(StringSoFactorType.STRING);

        private final int size;
        private final List<SoFactorType<?>> inputs;

        ConcatExtractorBase(final int size) {
            this.size = size;
            inputs = new ArrayList<>(size);
            for (int i = 0; i < size; ++i) {
                inputs.add(StringSoFactorType.STRING);
            }
        }

        @Override
        public void close() {
        }

        @Override
        public List<SoFactorType<?>> inputs() {
            return inputs;
        }

        @Override
        public List<SoFactorType<?>> outputs() {
            return OUTPUTS;
        }

        @Override
        public void registerInternals(
            final SoFactorsExtractorsRegistry registry)
            throws ConfigException
        {
            ConcatExtractorFactory.INSTANCE.registerInternals(registry);
        }

        protected abstract void processPart(StringBuilder sb, String part);

        @Override
        public void extract(
            final SoFactorsExtractorContext context,
            final SoFunctionInputs inputs,
            final FutureCallback<? super List<SoFactor<?>>> callback)
        {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < size; ++i) {
                String part = inputs.get(i, StringSoFactorType.STRING);
                if (part != null) {
                    processPart(sb, part);
                }
            }
            callback.completed(
                Collections.singletonList(
                    StringSoFactorType.STRING.createFactor(new String(sb))));
        }
    }

    private static class BareConcatExtractor extends ConcatExtractorBase {
        BareConcatExtractor(int size) {
            super(size);
        }

        @Override
        protected void processPart(
            final StringBuilder sb,
            final String part)
        {
            sb.append(part);
        }
    }

    private static class CharConcatExtractor extends ConcatExtractorBase {
        private final char delimiter;

        CharConcatExtractor(final int size, final char delimiter) {
            super(size);
            this.delimiter = delimiter;
        }

        @Override
        protected void processPart(
            final StringBuilder sb,
            final String part)
        {
            if (sb.length() > 0) {
                sb.append(delimiter);
            }
            sb.append(part);
        }
    }

    private static class ConcatExtractor extends ConcatExtractorBase {
        private final String delimiter;

        ConcatExtractor(final int size, final String delimiter) {
            super(size);
            this.delimiter = delimiter;
        }

        @Override
        protected void processPart(
            final StringBuilder sb,
            final String part)
        {
            if (sb.length() > 0) {
                sb.append(delimiter);
            }
            sb.append(part);
        }
    }
}

