package ru.yandex.mail.so.factors.extractors;

import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.DkimDomains;
import ru.yandex.mail.so.factors.types.DkimDomainsSoFactorType;
import ru.yandex.mail.so.factors.types.MailMetaSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.document.mail.MailMetaInfo;

public enum DkimDomainsExtractor implements SoFactorsExtractor {
    INSTANCE;

    private static final String NEEDLE = "\nx-yandex-authentication-results: ";
    private static final Pattern DKIM_HEADERS_PATTERN =
        Pattern.compile("(?:;|^)\\s*dkim=pass\\s+([^;]+)");
    private static final Pattern DKIM_DOMAIN_PATTERN =
        Pattern.compile("(?:header[.]i=[^@]*@|header[.]d=)(\\S+)");
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(MailMetaSoFactorType.MAIL_META);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(DkimDomainsSoFactorType.DKIM_DOMAINS);

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        MailMetaInfo meta = inputs.get(0, MailMetaSoFactorType.MAIL_META);
        if (meta == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        String headers = '\n' + meta.getLocal(MailMetaInfo.HEADERS) + '\n';
        int idx = headers.indexOf(NEEDLE);
        if (idx == -1) {
            context.logger().warning("Authentication results not found");
            callback.completed(NULL_RESULT);
            return;
        }
        int start = idx + NEEDLE.length();
        int end = headers.indexOf('\n', start);
        String authResults = headers.substring(start, end).trim();
        Matcher headersMatcher = DKIM_HEADERS_PATTERN.matcher(authResults);
        if (!headersMatcher.find()) {
            callback.completed(NULL_RESULT);
            return;
        }
        DkimDomains dkimDomains = new DkimDomains();
        do {
            String group = headersMatcher.group(1);
            Matcher domainMatcher = DKIM_DOMAIN_PATTERN.matcher(group);
            while (domainMatcher.find()) {
                dkimDomains.add(domainMatcher.group(1));
            }
        } while (headersMatcher.find());

        if (dkimDomains.isEmpty()) {
            callback.completed(NULL_RESULT);
        } else {
            callback.completed(
                Collections.singletonList(
                    DkimDomainsSoFactorType.DKIM_DOMAINS.createFactor(
                        dkimDomains)));
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        registry.typesRegistry().registerSoFactorType(
            MailMetaSoFactorType.MAIL_META);
        registry.typesRegistry().registerSoFactorType(
            DkimDomainsSoFactorType.DKIM_DOMAINS);
    }
}

