package ru.yandex.mail.so.factors.extractors;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.DkimDomainsSoFactorType;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.MailMetaSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.MailMetaInfo;

public class DkimStatsExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Arrays.asList(
            MailMetaSoFactorType.MAIL_META,
            DkimDomainsSoFactorType.DKIM_DOMAINS);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(JsonObjectSoFactorType.JSON_OBJECT);

    private final AsyncClient client;
    private final HttpHost host;

    public DkimStatsExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableHttpHostConfig hostConfig =
            new HttpHostConfigBuilder(config).build();
        client = context.asyncClientRegistrar().client(name, hostConfig);
        host = hostConfig.host();
    }

    @Override
    public void close() {
        // Client already handled by registrar
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        MailMetaInfo meta = inputs.get(0, MailMetaSoFactorType.MAIL_META);
        if (meta == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        String timestampStr =
            meta.getLocal(MailMetaInfo.GATEWAY_RECEIVED_DATE);
        long timestamp;
        try {
            timestamp = Long.valueOf(timestampStr);
        } catch (RuntimeException e) {
            callback.failed(e);
            return;
        }
        String from = meta.getSender();
        if (from == null) {
            context.logger().info("Sender address not found");
            callback.completed(NULL_RESULT);
            return;
        }
        Set<String> dkimDomains =
            inputs.get(1, DkimDomainsSoFactorType.DKIM_DOMAINS);
        if (dkimDomains == null) {
            dkimDomains = Collections.emptySet();
        }
        QueryConstructor query =
            new QueryConstructor("/api/async/so/get-dkim-stats?timestamp=");
        try {
            query.sb().append(timestamp);
            query.append("from", from);
            for (String dkimDomain: dkimDomains) {
                query.append("dkim-domain", dkimDomain);
            }
        } catch (BadRequestException e) {
            callback.failed(e);
            return;
        }
        AsyncClient client = this.client.adjust(context.httpContext());
        client.execute(
            host,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.requestsListener().createContextGeneratorFor(client),
            new Callback(callback));
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        DkimStatsExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SoFactor<?>>>
    {
        Callback(final FutureCallback<? super List<SoFactor<?>>> callback) {
            super(callback);
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                callback.completed(
                    Collections.singletonList(
                        JsonObjectSoFactorType.JSON_OBJECT.createFactor(
                            result.get("dkim_stats").asMap())));
            } catch (JsonException e) {
                failed(e);
            }
        }
    }
}

