package ru.yandex.mail.so.factors.extractors;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.wmi.FilterSearchErrorSuppressingFutureCallback;
import ru.yandex.client.wmi.FilterSearchJsonResponseConsumerFactory;
import ru.yandex.function.CorpUidPredicate;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.config.URIConfigBuilder;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.JsonMapSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class FilterSearchExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(JsonMapSoFactorType.JSON_MAP);

    private final AsyncClient client;
    private final String uri;
    private final AsyncClient corpClient;
    private final String corpUri;

    public FilterSearchExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableURIConfig uriConfig = new URIConfigBuilder(config).build();
        client = context.asyncClientRegistrar().client(name, uriConfig);
        uri =
            uriConfig.uri().toASCIIString()
            + uriConfig.firstCgiSeparator()
            + "full_folders_and_labels=1&uid=";

        ImmutableURIConfig corpConfig =
            new URIConfigBuilder(config.section("corp")).build();
        corpClient =
            context.asyncClientRegistrar().client("corp-" + name, corpConfig);
        corpUri =
            corpConfig.uri().toASCIIString()
            + corpConfig.firstCgiSeparator()
            + "full_folders_and_labels=1&uid=";
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return INPUTS;
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        FilterSearchExtractorFactory.INSTANCE.registerInternals(registry);
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        JsonMap change = inputs.get(0, JsonMapSoFactorType.JSON_MAP);
        if (change == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        try {
            long uid = change.getLong("uid");
            AsyncClient client;
            String uri;
            if (CorpUidPredicate.INSTANCE.test(uid)) {
                client = corpClient;
                uri = corpUri;
            } else {
                client = this.client;
                uri = this.uri;
            }
            StringBuilder sb = new StringBuilder(uri);
            sb.append(uid);
            JsonMap result = change.deepCopy();
            JsonList changed = result.remove("changed").asList();
            int size = changed.size();
            Map<String, JsonMap> midToChange = new HashMap<>(size << 1);
            for (int i = 0; i < size; ++i) {
                JsonMap singleChange = changed.get(i).asMap();
                String mid = singleChange.getString("mid");
                try {
                    Long.parseLong(mid);
                } catch (RuntimeException e) {
                    throw new BadRequestException(
                        "Malformed mid <" + mid + '>',
                        e);
                }
                midToChange.put(mid, singleChange);
                sb.append("&mids=");
                sb.append(mid);
            }
            Callback filterSearchCallback =
                new Callback(callback, result, midToChange);
            if (midToChange.isEmpty()) {
                filterSearchCallback.completed(JsonList.EMPTY);
            } else {
                client = client.adjust(context.httpContext());
                client.execute(
                    new AsyncGetURIRequestProducerSupplier(new String(sb)),
                    FilterSearchJsonResponseConsumerFactory.OK,
                    context.requestsListener()
                        .createContextGeneratorFor(client),
                    new IdempotentFutureCallback<>(
                        new FilterSearchErrorSuppressingFutureCallback<>(
                            filterSearchCallback,
                            JsonList.EMPTY,
                            context.logger())));
            }
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonList, List<SoFactor<?>>>
    {
        private final JsonMap result;
        private final Map<String, JsonMap> midToChange;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final JsonMap result,
            final Map<String, JsonMap> midToChange)
        {
            super(callback);
            this.result = result;
            this.midToChange = midToChange;
        }

        @Override
        public void completed(final JsonList envelopes) {
            try {
                int size = envelopes.size();
                JsonList changed =
                    new JsonList(BasicContainerFactory.INSTANCE, size);
                result.put("changed", changed);
                for (int i = 0; i < size; ++i) {
                    JsonMap envelope = envelopes.get(i).asMap();
                    String mid = envelope.getString("mid");
                    JsonMap singleChange = midToChange.remove(mid);
                    singleChange.put("envelope", envelope);
                    changed.add(singleChange);
                }
                callback.completed(
                    Collections.singletonList(
                        JsonMapSoFactorType.JSON_MAP.createFactor(result)));
            } catch (Exception e) {
                failed(e);
            }
        }
    }
}

