package ru.yandex.mail.so.factors.extractors;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.mail.so.factors.BasicSoFunctionInputs;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.JsonListSoFactorType;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class ForEachJsonObjectExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(JsonListSoFactorType.JSON_LIST);

    private final SoFactorsExtractor extractor;
    private final boolean parallel;
    private final int inputsSize;
    private final List<SoFactorType<?>> inputs;

    public ForEachJsonObjectExtractor(
        final List<SoFunctionArgumentInfo> inputs,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        String extractorName = config.getString("extractor");
        extractor = context.registry().getExtractor(extractorName);
        if (extractor == null) {
            throw new ConfigException(
                "Extractor <" + extractorName + "> not found");
        }
        parallel = config.getBoolean("parallel", false);

        SoFactorsExtractor.forceInputType(
            JsonListSoFactorType.JSON_LIST,
            inputs,
            0);
        List<SoFactorType<?>> extractorInputs = extractor.inputs();
        SoFactorsExtractor.forceInputsSize(extractorInputs.size(), inputs);
        SoFactorType<?> firstExtractorInput = extractorInputs.get(0);
        if (firstExtractorInput != JsonObjectSoFactorType.JSON_OBJECT) {
            throw new ConfigException(
                "Extractor first argument type expected to be "
                + JsonObjectSoFactorType.JSON_OBJECT
                + ", but was " + firstExtractorInput);
        }
        SoFactorsExtractor.forceOutputTypes(
            Collections.singletonList(JsonObjectSoFactorType.JSON_OBJECT),
            extractor.outputs());
        inputsSize = inputs.size();
        this.inputs = new ArrayList<>(inputsSize);
        this.inputs.add(JsonListSoFactorType.JSON_LIST);
        for (int i = 1; i < inputsSize; ++i) {
            SoFactorType<?> type = extractorInputs.get(i);
            SoFactorsExtractor.forceInputType(type, inputs, i);
            this.inputs.add(type);
        }
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        JsonList docs = inputs.get(0, JsonListSoFactorType.JSON_LIST);
        if (docs == null) {
            callback.completed(NULL_RESULT);
        } else {
            MultiFutureCallback<? super List<SoFactor<?>>> docsCallback =
                new MultiFutureCallback<>(
                    new Callback(callback, docs.containerFactory()));
            int size = docs.size();
            for (int i = 0; i < size; ++i) {
                JsonObject doc = docs.get(i);
                SoFactorsExtractorContext docContext =
                    new PrefixedSoFactorsExtractorContext(
                        context,
                        Integer.toString(i));
                List<SoFactor<?>> factors = new ArrayList<>(inputsSize);
                factors.add(
                    JsonObjectSoFactorType.JSON_OBJECT.createFactor(doc));
                for (int j = 1; j < inputsSize; ++j) {
                    factors.add(inputs.get(j));
                }
                BasicSoFunctionInputs docInputs =
                    new BasicSoFunctionInputs(
                        docContext.accessViolationHandler(),
                        factors);
                FutureCallback<? super List<SoFactor<?>>> docCallback =
                    docsCallback.newCallback();
                if (parallel) {
                    docContext.executor().execute(
                        new Task(
                            docContext,
                            docInputs,
                            docCallback,
                            extractor));
                } else {
                    extractor.extract(docContext, docInputs, docCallback);
                }
            }
            docsCallback.done();
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        ForEachJsonObjectExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class Task implements Runnable {
        private final SoFactorsExtractorContext context;
        private final SoFunctionInputs inputs;
        private final FutureCallback<? super List<SoFactor<?>>> callback;
        private final SoFactorsExtractor extractor;

        Task(
            final SoFactorsExtractorContext context,
            final SoFunctionInputs inputs,
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final SoFactorsExtractor extractor)
        {
            this.context = context;
            this.inputs = inputs;
            this.callback = callback;
            this.extractor = extractor;
        }

        @Override
        public void run() {
            extractor.extract(context, inputs, callback);
        }
    }

    private static class Callback extends AbstractFilterFutureCallback<
        List<List<SoFactor<?>>>,
        List<SoFactor<?>>>
    {
        private final ContainerFactory containerFactory;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final ContainerFactory containerFactory)
        {
            super(callback);
            this.containerFactory = containerFactory;
        }

        @Override
        public void completed(final List<List<SoFactor<?>>> results) {
            int resultsSize = results.size();
            JsonList list = new JsonList(containerFactory, resultsSize);
            for (int i = 0; i < resultsSize; ++i) {
                List<SoFactor<?>> result = results.get(i);
                int resultSize = result.size();
                if (resultSize == 1) {
                    SoFactor<?> factor = result.get(0);
                    if (factor != null
                        && factor.value() != null
                        && factor.type() == JsonObjectSoFactorType.JSON_OBJECT)
                    {
                        list.add(
                            JsonObjectSoFactorType.JSON_OBJECT.cast(
                                factor.value()));
                    }
                }
            }
            callback.completed(
                Collections.singletonList(
                    JsonListSoFactorType.JSON_LIST.createFactor(list)));
        }
    }
}

