package ru.yandex.mail.so.factors.extractors;

import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.so.factors.BasicSoFunctionInputs;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocSoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocsSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class ForEachTikaiteDocExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(TikaiteDocsSoFactorType.TIKAITE_DOCS);
    private static final List<SoFactorType<?>> EXTRACTOR_TYPES =
        Collections.singletonList(TikaiteDocSoFactorType.TIKAITE_DOC);

    private final SoFactorsExtractor extractor;
    private final boolean parallel;

    public ForEachTikaiteDocExtractor(
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        String extractorName = config.getString("extractor");
        extractor = context.registry().getExtractor(extractorName);
        if (extractor == null) {
            throw new ConfigException(
                "Extractor <" + extractorName + "> not found");
        }
        parallel = config.getBoolean("parallel", false);

        SoFactorsExtractor.forceOutputTypes(
            EXTRACTOR_TYPES,
            extractor.inputs());
        SoFactorsExtractor.forceOutputTypes(
            EXTRACTOR_TYPES,
            extractor.outputs());
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return INPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        JsonList docs = inputs.get(0, TikaiteDocsSoFactorType.TIKAITE_DOCS);
        if (docs == null) {
            callback.completed(NULL_RESULT);
        } else {
            MultiFutureCallback<? super List<SoFactor<?>>> docsCallback =
                new MultiFutureCallback<>(
                    new Callback(callback, docs.containerFactory()));
            int size = docs.size();
            int i = 0;
            try {
                for (; i < size; ++i) {
                    JsonMap doc = docs.get(i).asMap();
                    SoFactorsExtractorContext partContext =
                        new PrefixedSoFactorsExtractorContext(
                            context,
                            "hid:" + doc.getString("hid", null));
                    FutureCallback<? super List<SoFactor<?>>> docCallback =
                        docsCallback.newCallback();
                    BasicSoFunctionInputs docInputs =
                        new BasicSoFunctionInputs(
                            partContext.accessViolationHandler(),
                            TikaiteDocSoFactorType.TIKAITE_DOC.createFactor(
                                doc));
                    if (parallel) {
                        partContext.executor().execute(
                            new Task(
                                partContext,
                                docInputs,
                                docCallback,
                                extractor));
                    } else {
                        extractor.extract(partContext, docInputs, docCallback);
                    }
                }
                docsCallback.done();
            } catch (JsonException e) {
                callback.failed(
                    new IllegalArgumentException(
                        "Failed to process doc #" + i + " in:\n"
                        + JsonType.HUMAN_READABLE.toString(docs)));
            }
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        ForEachTikaiteDocExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class Task implements Runnable {
        private final SoFactorsExtractorContext context;
        private final SoFunctionInputs inputs;
        private final FutureCallback<? super List<SoFactor<?>>> callback;
        private final SoFactorsExtractor extractor;

        Task(
            final SoFactorsExtractorContext context,
            final SoFunctionInputs inputs,
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final SoFactorsExtractor extractor)
        {
            this.context = context;
            this.inputs = inputs;
            this.callback = callback;
            this.extractor = extractor;
        }

        @Override
        public void run() {
            extractor.extract(context, inputs, callback);
        }
    }

    private static class Callback extends AbstractFilterFutureCallback<
        List<List<SoFactor<?>>>,
        List<SoFactor<?>>>
    {
        private final ContainerFactory containerFactory;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final ContainerFactory containerFactory)
        {
            super(callback);
            this.containerFactory = containerFactory;
        }

        @Override
        public void completed(final List<List<SoFactor<?>>> results) {
            int resultsSize = results.size();
            JsonList list = new JsonList(containerFactory, resultsSize);
            for (int i = 0; i < resultsSize; ++i) {
                List<SoFactor<?>> result = results.get(i);
                int resultSize = result.size();
                if (resultSize == 1) {
                    SoFactor<?> factor = result.get(0);
                    if (factor != null
                        && factor.type() == TikaiteDocSoFactorType.TIKAITE_DOC)
                    {
                        list.add(
                            TikaiteDocSoFactorType.TIKAITE_DOC.cast(
                                factor.value()));
                    }
                }
            }
            callback.completed(
                Collections.singletonList(
                    TikaiteDocsSoFactorType.TIKAITE_DOCS.createFactor(list)));
        }
    }
}

