package ru.yandex.mail.so.factors.extractors;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.config.URIConfigBuilder;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.ByteArrayProcessableAsyncConsumerFactory;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;

public class HttpGetExtractor implements SoFactorsExtractor {
    private final List<SoFactorType<?>> inputs;
    private final List<SoFactorType<?>> outputs;
    private final UriConstructor uriConstructor;
    private final Requester<?> requester;

    public HttpGetExtractor(
        final String name,
        final List<SoFunctionArgumentInfo> inputs,
        final List<SoFactorType<?>> outputs,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableURIConfig uriConfig = new URIConfigBuilder(config).build();
        AsyncClient client =
            context.asyncClientRegistrar().client(name, uriConfig);
        EnumSet<RequestErrorType> suppressErrorTypes =
            config.get(
                "suppress-error-types",
                EnumSet.noneOf(RequestErrorType.class),
                new CollectionParser<>(
                    NonEmptyValidator.TRIMMED.andThen(
                        new EnumParser<>(RequestErrorType.class)),
                    () -> EnumSet.noneOf(RequestErrorType.class)));
        String uri = uriConfig.uri().toASCIIString();
        int inputsSize = inputs.size();
        if (inputsSize == 0) {
            uriConstructor = new IdentityUriConstructor(uri);
        } else if (inputsSize == 1) {
            uriConstructor = new SimpleUriConstructor(uri);
        } else if ((inputsSize & 1) == 0) {
            uriConstructor =
                new PairsUriConstructor(
                    uri + uriConfig.firstCgiSeparator(),
                    inputsSize);
        } else {
            throw new ConfigException(
                "HttpGetExtractor accepts either 1 "
                + "or even number of arguments");
        }
        this.inputs = new ArrayList<>(inputsSize);
        for (int i = 0; i < inputsSize; ++i) {
            this.inputs.add(StringSoFactorType.STRING);
        }
        if (outputs.size() != 1) {
            throw new ConfigException("Exactly one output expected");
        }
        SoFactorType<?> output = outputs.get(0);
        this.outputs = Collections.singletonList(output);
        if (output == BinarySoFactorType.BINARY) {
            requester =
                new Requester<>(
                    client,
                    suppressErrorTypes,
                    ByteArrayProcessableAsyncConsumerFactory.OK,
                    BinarySoFactorType.BINARY);
        } else if (output == JsonObjectSoFactorType.JSON_OBJECT) {
            requester =
                new Requester<>(
                    client,
                    suppressErrorTypes,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    JsonObjectSoFactorType.JSON_OBJECT);
        } else if (output == StringSoFactorType.STRING) {
            requester =
                new Requester<>(
                    client,
                    suppressErrorTypes,
                    AsyncStringConsumerFactory.OK,
                    StringSoFactorType.STRING);
        } else {
            throw new ConfigException(
                "Output factor type not supported: " + output);
        }
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return outputs;
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        HttpGetExtractorFactory.INSTANCE.registerInternals(registry);
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        try {
            String uri = uriConstructor.uriFor(inputs, context);
            if (uri == null) {
                callback.completed(NULL_RESULT);
                return;
            }
            requester.sendRequest(uri, context, callback);
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    private interface UriConstructor {
        // Returns null if any of inputs is null
        String uriFor(
            SoFunctionInputs inputs,
            SoFactorsExtractorContext context)
            throws HttpException;
    }

    private static class IdentityUriConstructor implements UriConstructor {
        private final String uri;

        IdentityUriConstructor(final String uri) {
            this.uri = uri;
        }

        @Override
        public String uriFor(
            final SoFunctionInputs inputs,
            final SoFactorsExtractorContext context)
        {
            return uri;
        }
    }

    private static class SimpleUriConstructor implements UriConstructor {
        private final String uri;

        SimpleUriConstructor(final String uri) {
            this.uri = uri;
        }

        @Override
        public String uriFor(
            final SoFunctionInputs inputs,
            final SoFactorsExtractorContext context)
            throws HttpException
        {
            String input = inputs.get(0, StringSoFactorType.STRING);
            if (input == null) {
                context.logger().warning("Missing input");
                return null;
            }
            QueryConstructor query = new QueryConstructor(uri);
            query.append(input);
            return query.toString();
        }
    }

    private static class PairsUriConstructor implements UriConstructor {
        private final String uri;
        private final int inputsSize;

        PairsUriConstructor(final String uri, final int inputsSize) {
            this.uri = uri;
            this.inputsSize = inputsSize;
        }

        @Override
        public String uriFor(
            final SoFunctionInputs inputs,
            final SoFactorsExtractorContext context)
            throws HttpException
        {
            QueryConstructor query = new QueryConstructor(uri, false);
            for (int i = 0; i < inputsSize; ++i) {
                String name = inputs.get(i, StringSoFactorType.STRING);
                if (name == null) {
                    context.logger().warning(
                        "Missing parameter name at arg #" + i);
                    return null;
                }
                String value = inputs.get(++i, StringSoFactorType.STRING);
                if (value == null) {
                    context.logger().warning(
                        "Missing parameter value at arg #" + i);
                    return null;
                }
                query.append(name, value);
            }
            return query.toString();
        }
    }

    private static class Requester<T> {
        private final AsyncClient client;
        private final EnumSet<RequestErrorType> suppressErrorTypes;
        private final HttpAsyncResponseConsumerFactory<T> consumerFactory;
        private final SoFactorType<T> factorType;

        Requester(
            final AsyncClient client,
            final EnumSet<RequestErrorType> suppressErrorTypes,
            final HttpAsyncResponseConsumerFactory<T> consumerFactory,
            final SoFactorType<T> factorType)
        {
            this.client = client;
            this.suppressErrorTypes = suppressErrorTypes;
            this.consumerFactory = consumerFactory;
            this.factorType = factorType;
        }

        public void sendRequest(
            final String uri,
            final SoFactorsExtractorContext context,
            final FutureCallback<? super List<SoFactor<?>>> callback)
            throws URISyntaxException
        {
            AsyncClient client = this.client.adjust(context.httpContext());
            client.execute(
                new AsyncGetURIRequestProducerSupplier(uri),
                consumerFactory,
                context.requestsListener().createContextGeneratorFor(client),
                new Callback<>(callback, factorType, suppressErrorTypes));
        }
    }

    private static class Callback<T>
        extends AbstractFilterFutureCallback<T, List<SoFactor<?>>>
    {
        private final SoFactorType<T> factorType;
        private final EnumSet<RequestErrorType> suppressErrorTypes;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final SoFactorType<T> factorType,
            final EnumSet<RequestErrorType> suppressErrorTypes)
        {
            super(callback);
            this.factorType = factorType;
            this.suppressErrorTypes = suppressErrorTypes;
        }

        @Override
        public void completed(final T result) {
            callback.completed(
                Collections.singletonList(factorType.createFactor(result)));
        }

        @Override
        public void failed(final Exception e) {
            if (!suppressErrorTypes.isEmpty()
                && suppressErrorTypes.contains(
                    RequestErrorType.ERROR_CLASSIFIER.apply(e)))
            {
                callback.completed(NULL_RESULT);
            } else {
                callback.failed(e);
            }
        }
    }
}

