package ru.yandex.mail.so.factors.extractors;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.JsonMapSoFactorType;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocSoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocsSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class LogExtractor implements SoFactorsExtractor {
    private final int size;
    private final List<SoFactorType<?>> inputs;
    private final List<SoFactorType<?>> outputs;
    private final JsonType jsonType;
    private final boolean[] isStringInput;
    private final String[] prefixes;

    public LogExtractor(
        final List<SoFunctionArgumentInfo> inputs,
        final IniConfig config)
        throws ConfigException
    {
        size = inputs.size();
        if (size < 2) {
            throw new ConfigException(
                "At least two inputs required, got: " + inputs);
        }
        SoFactorType<?> firstInputType = inputs.get(0).type();
        this.outputs = Collections.singletonList(firstInputType);
        isStringInput = new boolean[size];
        prefixes = new String[size];
        this.inputs = new ArrayList<>(size);
        this.inputs.add(firstInputType);
        jsonType = JsonTypeExtractor.NORMAL.extract(config);
        for (int i = 1; i < size; ++i) {
            SoFunctionArgumentInfo input = inputs.get(i);
            SoFactorType<?> type = input.type();
            if (type == JsonMapSoFactorType.JSON_MAP
                || type == JsonObjectSoFactorType.JSON_OBJECT
                || type == TikaiteDocSoFactorType.TIKAITE_DOC
                || type == TikaiteDocsSoFactorType.TIKAITE_DOCS)
            {
                this.inputs.add(type);
            } else {
                SoFactorsExtractor.forceInputType(
                    StringSoFactorType.STRING,
                    inputs,
                    i);
                this.inputs.add(type);
                isStringInput[i] = true;
            }
            prefixes[i] = config.getString(input.name());
        }
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return outputs;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        SoFactor<?> firstFactor = inputs.get(0);
        if (firstFactor == null) {
            callback.completed(NULL_RESULT);
        } else {
            callback.completed(Collections.singletonList(firstFactor));
        }

        for (int i = 1; i < size; ++i) {
            SoFactor<?> factor = inputs.get(i);
            if (factor != null) {
                if (isStringInput[i]) {
                    String value =
                        StringSoFactorType.STRING.cast(factor.value());
                    if (value == null) {
                        context.accessViolationHandler()
                            .handleFactorTypeMismatch(
                                i,
                                StringSoFactorType.STRING,
                                factor);
                    } else if (context.logger().isLoggable(Level.INFO)) {
                        context.logger().info(prefixes[i] + value);
                    }
                } else {
                    JsonObject value = JsonObjectSoFactorType.JSON_OBJECT.cast(
                        factor.value());
                    if (value == null) {
                        context.accessViolationHandler()
                            .handleFactorTypeMismatch(
                                i,
                                JsonObjectSoFactorType.JSON_OBJECT,
                                factor);
                    } else if (context.logger().isLoggable(Level.INFO)) {
                        StringBuilder sb = new StringBuilder(prefixes[i]);
                        jsonType.toStringBuilder(sb, value);
                        context.logger().info(new String(sb));
                    }
                }
            }
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        LogExtractorFactory.INSTANCE.registerInternals(registry);
    }
}

