package ru.yandex.mail.so.factors.extractors;

import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.digest.Fnv;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.ByteArrayProcessor;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;

public enum MailFnv64Extractor implements SoFactorsExtractor {
    INSTANCE;

    private static final int MAX_HASHES = 1024;

    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(BinarySoFactorType.RAW_MAIL);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(StringSoFactorType.STRING);

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        ByteArrayProcessable rawMail =
            inputs.get(0, BinarySoFactorType.RAW_MAIL);
        if (rawMail == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        String hashes = rawMail.processWith(new HashesBuilder());
        if (hashes.isEmpty()) {
            callback.completed(NULL_RESULT);
        } else {
            callback.completed(
                Collections.singletonList(
                    StringSoFactorType.STRING.createFactor(hashes)));
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        registry.typesRegistry().registerSoFactorType(
            BinarySoFactorType.RAW_MAIL);
        registry.typesRegistry().registerSoFactorType(
            StringSoFactorType.STRING);
    }

    private static class HashesBuilder
        implements ByteArrayProcessor<String, RuntimeException>
    {
        private final StringBuilder sb = new StringBuilder();
        private boolean prevEmpty = true;
        private int hashes = 0;

        private void processLine(
            final byte[] buf,
            int lineStart,
            int lineEnd)
        {
            int initialLen = lineEnd - lineStart;
            while (lineStart < lineEnd
                && Character.isWhitespace(buf[lineStart]))
            {
                ++lineStart;
            }
            while (lineStart < lineEnd
                && Character.isWhitespace(buf[lineEnd - 1]))
            {
                --lineEnd;
            }
            int len = lineEnd - lineStart;
            if (len == 0) {
                if (!prevEmpty) {
                    sb.append('\n');
                    prevEmpty = true;
                }
            } else {
                long hash = Fnv.fnv64(buf, lineStart, len);
                sb.append(hashes++);
                sb.append('\t');
                sb.append(initialLen);
                sb.append('\t');
                sb.append(len);
                sb.append('\t');
                sb.append(Long.toHexString(hash));
                sb.append('\n');
                prevEmpty = false;
            }
        }

        @Override
        public String process(final byte[] buf, final int off, final int len) {
            int pos = off;
            int end = off + len;
            while (pos < end) {
                if (buf[pos++] == '\n') {
                    if (pos < end) {
                        if (buf[pos] == '\r') {
                            ++pos;
                            if (pos >= end) {
                                return "";
                            }
                        }
                        if (buf[pos++] == '\n') {
                            break;
                        }
                    }
                }
            }
            if (pos >= end) {
                return "";
            }
            int prev = pos;
            while (pos < end && hashes < MAX_HASHES) {
                if (buf[pos++] == '\n') {
                    int lineStart = prev;
                    int lineEnd = pos - 1;
                    processLine(buf, lineStart, lineEnd);
                    prev = pos;
                }
            }
            processLine(buf, prev, end);
            return new String(sb);
        }
    }
}

