package ru.yandex.mail.so.factors.extractors;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.ObjectsStorage;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class NarrowExtractor implements SoFactorsExtractor, Stater {
    private final AtomicLong extractionsAccepted = new AtomicLong();
    private final AtomicLong extractionsRejected = new AtomicLong();
    private final List<SoFactorType<?>> inputs;
    private final List<SoFactorType<?>> outputs;
    private final ObjectsStorage<SoFactorsExtractor> extractors;
    private final int limit;
    private final String statsPrefix;
    private final String chartsCategory;
    private final String chartsTitle;
    private final String acceptedSignal;
    private final String rejectedSignal;
    private final String availableSignal;
    private final String limitSignal;
    private final List<SoFactor<?>> nullResult;

    public NarrowExtractor(
        final List<SoFactorType<?>> inputs,
        final List<SoFactorType<?>> outputs,
        final ObjectsStorage<SoFactorsExtractor> extractors,
        final int limit,
        final String statsPrefix,
        final String chartsCategory,
        final String chartsTitle)
    {
        this.inputs = inputs;
        this.outputs = outputs;
        this.extractors = extractors;
        this.limit = limit;
        this.statsPrefix = statsPrefix;
        this.chartsCategory = chartsCategory;
        this.chartsTitle = chartsTitle;
        acceptedSignal = statsPrefix + "-extractions-accepted_dmmm";
        rejectedSignal = statsPrefix + "-extractions-rejected_dmmm";
        availableSignal = statsPrefix + "-extractors-available_ammm";
        limitSignal = statsPrefix + "-extractors-limit_ammm";
        int outputsSize = outputs().size();
        nullResult = new ArrayList<>(outputsSize);
        for (int i = 0; i < outputsSize; ++i) {
            nullResult.add(null);
        }
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return outputs;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        SoFactorsExtractor extractor = extractors.get();
        if (extractor == null) {
            extractionsRejected.incrementAndGet();
            callback.completed(nullResult);
        } else {
            extractionsAccepted.incrementAndGet();
            extractor.extract(
                context,
                inputs,
                new IdempotentFutureCallback<>(
                    new Callback(callback, extractors, extractor)));
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        if (statsPrefix != null) {
            registry.statersRegistrar().registerStater(this);
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        statsConsumer.stat(acceptedSignal, extractionsAccepted.get());
        statsConsumer.stat(rejectedSignal, extractionsRejected.get());
        statsConsumer.stat(availableSignal, extractors.size());
        statsConsumer.stat(limitSignal, limit);
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix)
    {
        String chartsPrefix = statsPrefix + this.statsPrefix;
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChart chart = new GolovanChart(
            "-accepted-rate",
            " accepted tasks rate",
            false,
            false,
            0d);
        chart.addSplitSignal(
            config,
            "perc(" + statsPrefix + acceptedSignal
            + ",sum(" + statsPrefix + acceptedSignal
            + ',' + statsPrefix + rejectedSignal + ')' + ')',
            1,
            true,
            false);
        group.addChart(chart);

        chart = new GolovanChart(
            "-extractors-available",
            " available extractors",
            false,
            false,
            0d);
        chart.addSignal(
            new GolovanSignal(
                statsPrefix + availableSignal,
                config.tag(),
                "available",
                null,
                1,
                false));
        chart.addSignal(
            new GolovanSignal(
                statsPrefix + limitSignal,
                config.tag(),
                "limit",
                "#ff0000",
                1,
                false));
        group.addChart(chart);

        panel.addCharts(chartsCategory, chartsTitle, group);
    }

    private static class Callback
        extends FilterFutureCallback<List<SoFactor<?>>>
    {
        private final ObjectsStorage<SoFactorsExtractor> extractors;
        private final SoFactorsExtractor extractor;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final ObjectsStorage<SoFactorsExtractor> extractors,
            final SoFactorsExtractor extractor)
        {
            super(callback);
            this.extractors = extractors;
            this.extractor = extractor;
        }

        @Override
        public void cancelled() {
            extractors.put(extractor);
            super.cancelled();
        }

        @Override
        public void completed(final List<SoFactor<?>> result) {
            extractors.put(extractor);
            super.completed(result);
        }

        @Override
        public void failed(final Exception e) {
            extractors.put(extractor);
            super.failed(e);
        }
    }
}

