package ru.yandex.mail.so.factors.extractors;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.concurrent.ConcurrentStackStorage;
import ru.yandex.concurrent.ObjectsStorage;
import ru.yandex.mail.so.factors.SoFunctionArgumentInfo;
import ru.yandex.mail.so.factors.dsl.DslParser;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.util.timesource.TimeSource;

public enum NarrowExtractorFactory
    implements TypeCheckedSoFactorsExtractorFactory
{
    INSTANCE;

    private static final Pattern BLOCK_START =
        Pattern.compile("^\\s*[{]\\s*\n");
    private static final Pattern BLOCK_END =
        Pattern.compile("\n\\s*[}]\\s*\n");

    @Override
    public void close() {
    }

    @Override
    public SoFactorsExtractor createExtractorUnchecked(
        final String name,
        final List<SoFunctionArgumentInfo> inputs,
        final List<SoFactorType<?>> outputs,
        final SoFactorsExtractorFactoryContext context,
        final String body)
        throws ConfigException, IOException
    {
        NarrowExtractorBase base =
            new NarrowExtractorBase(name, inputs, outputs, context, body);
        if (base.queued) {
            return new QueuedNarrowExtractor(
                base.inputs,
                base.outputs,
                base.extractors,
                base.limit,
                base.statsPrefix,
                base.chartsCategory,
                base.chartsTitle);
        } else {
            return new NarrowExtractor(
                base.inputs,
                base.outputs,
                base.extractors,
                base.limit,
                base.statsPrefix,
                base.chartsCategory,
                base.chartsTitle);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry) {
    }

    private static class NarrowExtractorBase {
        private final ObjectsStorage<SoFactorsExtractor> extractors =
            new ConcurrentStackStorage<>();
        private final int limit;
        private final boolean queued;
        private final String statsPrefix;
        private final String chartsCategory;
        private final String chartsTitle;
        private final List<SoFactorType<?>> inputs;
        private final List<SoFactorType<?>> outputs;

        public NarrowExtractorBase(
            final String name,
            final List<SoFunctionArgumentInfo> inputs,
            final List<SoFactorType<?>> outputs,
            final SoFactorsExtractorFactoryContext context,
            final String body)
            throws ConfigException, IOException
        {
            Matcher matcher = BLOCK_START.matcher(body);
            if (!matcher.find()) {
                throw new ConfigException("'{' expected");
            }
            int blockStart = matcher.end();
            matcher = BLOCK_END.matcher(body);
            if (!matcher.find(blockStart)) {
                throw new ConfigException("'}' expected");
            }
            String extractorBody = body.substring(matcher.end());
            IniConfig config =
                context.readBodyAsIniConfig(
                    body.substring(blockStart, matcher.start()));
            limit = config.get("limit", PositiveIntegerValidator.INSTANCE);
            queued = config.getBoolean("queued");
            String extractorName = config.getString("extractor");
            SoFactorsExtractor extractor =
                context.registry().getExtractor(extractorName);
            if (extractor == null) {
                SoFactorsExtractorFactory factory =
                    context.registry().getExtractorFactory(extractorName);
                if (factory == null) {
                    for (int i = 0; i < limit; ++i) {
                        context.logger().info(
                            "Creating extractor " + name
                            + ' ' + '#' + i + " with entry point "
                            + extractorName);
                        long start = TimeSource.INSTANCE.currentTimeMillis();
                        SoFactorsExtractorsRegistry localRegistry =
                            new SoFactorsExtractorsRegistry(context.registry());
                        new DslParser(
                            new SoFactorsExtractorFactoryContext(
                                context.path(),
                                localRegistry,
                                context.statsConsumerFactory(),
                                context.violationsCounter(),
                                context.luaErrorsCounter(),
                                context.threadGroup(),
                                context.asyncClientRegistrar(),
                                context.externalDataProvider(),
                                context.modulesCache(),
                                context.luaModulesCache(),
                                context.logger().addPrefix(name),
                                context.samplesNotifier(),
                                context.metricsTimeFrame()))
                            .parse(
                                new BufferedReader(
                                    new StringReader(extractorBody)));
                        context.registry().addNestedRegistry(localRegistry);
                        long end = TimeSource.INSTANCE.currentTimeMillis();
                        context.logger().info(
                            "Extractor " + name
                            + " created in " + (end - start) + " ms");
                        extractor = localRegistry.getExtractor(extractorName);
                        if (extractor == null) {
                            throw new ConfigException(
                                "Extractor " + extractorName + " not found");
                        }
                        SoFactorsExtractor.forceInputTypes(
                            extractor.inputs(),
                            inputs);
                        SoFactorsExtractor.forceOutputTypes(
                            extractor.outputs(),
                            outputs);
                        extractors.put(extractor);
                    }
                } else {
                    for (int i = 0; i < limit; ++i) {
                        context.logger().info(
                            "Creating extractor " + name
                            + ' ' + '#' + i + " with "
                            + factory.getClass().getSimpleName());
                        long start = TimeSource.INSTANCE.currentTimeMillis();
                        extractor = factory.createExtractor(
                            name,
                            inputs,
                            outputs,
                            context,
                            extractorBody);
                        context.registry().addAnonymousExtractor(extractor);
                        long end = TimeSource.INSTANCE.currentTimeMillis();
                        context.logger().info(
                            "Extractor " + name
                            + " created in " + (end - start) + " ms");
                        extractors.put(extractor);
                    }
                }
            } else {
                for (int i = 0; i < limit; ++i) {
                    extractors.put(extractor);
                }
                SoFactorsExtractor.forceEmptyBody(extractorBody);
                SoFactorsExtractor.forceInputTypes(extractor.inputs(), inputs);
                SoFactorsExtractor.forceOutputTypes(
                    extractor.outputs(),
                    outputs);
            }
            statsPrefix =
                config.get("stats-prefix", null, NonEmptyValidator.TRIMMED);
            chartsCategory =
                config.get("charts-category", null, NonEmptyValidator.TRIMMED);
            chartsTitle =
                config.get("charts-title", null, NonEmptyValidator.TRIMMED);
            config.checkUnusedKeys();
            this.inputs = extractor.inputs();
            this.outputs = extractor.outputs();
        }
    }
}

