package ru.yandex.mail.so.factors.extractors;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.BodyDescriptor;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.MimeTokenStream;
import org.apache.james.mime4j.stream.RecursionMode;
import org.apache.james.mime4j.util.MimeUtil;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.mail.mime.BodyDecoder;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;

public enum NestedMailExtractor implements SoFactorsExtractor {
    INSTANCE;

    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(BinarySoFactorType.RAW_MAIL);

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return INPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        ByteArrayProcessable rawMail =
            inputs.get(0, BinarySoFactorType.RAW_MAIL);
        if (rawMail == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        MimeTokenStream tokenStream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        tokenStream.setRecursionMode(RecursionMode.M_NO_RECURSE);
        try (ByteArrayInputStream in = rawMail.content()) {
            tokenStream.parse(in);
            EntityState state = tokenStream.getState();
            while (state != EntityState.T_END_OF_STREAM) {
                if (state == EntityState.T_BODY) {
                    BodyDescriptor bd = tokenStream.getBodyDescriptor();
                    if (MimeUtil.isMessage(bd.getMimeType())) {
                        ByteArrayProcessable result;
                        try (InputStream is = BodyDecoder.INSTANCE.apply(
                                tokenStream.getInputStream(),
                                bd.getTransferEncoding()))
                        {
                            result = IOStreamUtils
                                .consume(is)
                                .toByteArrayProcessable();
                        }
                        if (context.logger().isLoggable(Level.INFO)) {
                            context.logger().info(
                                "Nested mail length: " + result.length());
                        }
                        callback.completed(
                            Collections.singletonList(
                                BinarySoFactorType.RAW_MAIL.createFactor(
                                    result)));
                        return;
                    }
                }
                state = tokenStream.next();
            }
        } catch (IOException | MimeException e) {
            callback.failed(e);
            return;
        }
        callback.completed(NULL_RESULT);
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        registry.typesRegistry().registerSoFactorType(
            BinarySoFactorType.RAW_MAIL);
    }
}

