package ru.yandex.mail.so.factors.extractors;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.protobuf.ByteString;
import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.charset.Decoder;
import ru.yandex.http.config.ExternalDataConfigBuilder;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.server.ExternalDataSubscriber;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.api.v1.SmtpEnvelope;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.BooleanSoFactorType;
import ru.yandex.mail.so.factors.types.LongSoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.util.ip.Cidr;

public class OrgSettingsExtractor
    implements ExternalDataSubscriber, SoFactorsExtractor
{
    private static final List<SoFactorType<?>> INPUTS =
        Arrays.asList(
            SmtpEnvelopeSoFactorType.SMTP_ENVELOPE,
            LongSoFactorType.LONG,
            LongSoFactorType.LONG);
    private static final List<SoFactorType<?>> OUTPUTS =
        Arrays.asList(
            BooleanSoFactorType.BOOLEAN,
            LongSoFactorType.LONG);
    private static final Cidr[] EMPTY_IP_WHITELIST = new Cidr[0];

    private volatile OrgsSettings orgsSettings;

    public OrgSettingsExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        try {
            context.externalDataProvider().addExternalDataUpdater(
                "cretur",
                new ExternalDataConfigBuilder(config).build(),
                this);
        } catch (HttpException | IOException e) {
            throw new ConfigException(
                "Failed to fetch organizations settings",
                e);
        }
    }

    @Override
    public void updateExternalData(
        final ByteArrayProcessableWithContentType data)
        throws HttpException, IOException
    {
        orgsSettings = new OrgsSettings(data);
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        SmtpEnvelopeHolder envelope =
            inputs.get(0, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
        Long mailfromOrgId = inputs.get(1, LongSoFactorType.LONG);
        Long commonOrgId = inputs.get(2, LongSoFactorType.LONG);

        Boolean emailFromOrgWhitelist = null;
        if (envelope != null && commonOrgId != null) {
            OrgSettings settings = orgsSettings.orgs.get(commonOrgId);
            if (settings != null) {
                SmtpEnvelope smtpEnvelope = envelope.envelope();
                if (smtpEnvelope.hasConnectInfo()) {
                    ByteString address =
                        smtpEnvelope.getConnectInfo().getRemoteIp();
                    if (address != null) {
                        byte[] addr = address.toByteArray();
                            emailFromOrgWhitelist = Boolean.FALSE;
                            for (Cidr cidr: settings.ipWhitelist) {
                                if (cidr.matches(addr)) {
                                    emailFromOrgWhitelist = Boolean.TRUE;
                                    break;
                                }
                            }
                        }
                    }
            }
        }

        Long dailyLimit = null;
        if (mailfromOrgId != null) {
            OrgSettings settings = orgsSettings.orgs.get(mailfromOrgId);
            if (settings != null) {
                dailyLimit = settings.dailyLimit;
            }
        }

        List<SoFactor<?>> factors = new ArrayList<>(2);
        if (emailFromOrgWhitelist == null) {
            factors.add(null);
        } else {
            factors.add(
                BooleanSoFactorType.BOOLEAN.createFactor(
                    emailFromOrgWhitelist));
        }
        if (dailyLimit == null) {
            factors.add(null);
        } else {
            factors.add(LongSoFactorType.LONG.createFactor(dailyLimit));
        }

        callback.completed(factors);
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        OrgSettingsExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class OrgSettings {
        private final Long dailyLimit;
        private final Cidr[] ipWhitelist;

        OrgSettings(final Long dailyLimit, final Cidr[] ipWhitelist) {
            this.dailyLimit = dailyLimit;
            this.ipWhitelist = ipWhitelist;
        }
    }

    private static class OrgsSettings {
        // TODO: use sorted keys array for better compression
        private final Map<Long, OrgSettings> orgs;

        OrgsSettings(final ByteArrayProcessableWithContentType data)
            throws IOException, HttpException
        {
            Decoder decoder = new Decoder(data.contentType().getCharset());
            data.data().processWith(decoder);
            try {
                JsonMap map =
                    TypesafeValueContentHandler.parse(decoder).get("settings")
                        .asMap();
                orgs = new HashMap<>(map.size() << 1);
                for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
                    String orgId = entry.getKey();
                    JsonMap settings = entry.getValue().asMap();
                    Long dailyLimit = settings.getLong("daily_limit", null);
                    JsonList ipWhitelist =
                        settings.get("ip_whitelist").asListOrNull();
                    Cidr[] cidrs;
                    if (ipWhitelist == null) {
                        cidrs = EMPTY_IP_WHITELIST;
                    } else {
                        int size = ipWhitelist.size();
                        cidrs = new Cidr[size];
                        for (int i = 0; i < size; ++i) {
                            cidrs[i] =
                                Cidr.fromString(ipWhitelist.get(i).asString());
                        }
                    }
                    orgs.put(
                        Long.valueOf(orgId),
                        new OrgSettings(dailyLimit, cidrs));
                }
            } catch (JsonException | RuntimeException e) {
                throw new ServiceUnavailableException(
                    "Failed to parse json <" + decoder + '>',
                    e);
            }
        }
    }
}

