package ru.yandex.mail.so.factors.extractors;

import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.ObjectsStorage;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class QueuedNarrowExtractor implements SoFactorsExtractor, Stater {
    private final Queue<Task> tasks = new ConcurrentLinkedQueue<>();
    private final List<SoFactorType<?>> inputs;
    private final List<SoFactorType<?>> outputs;
    private final ObjectsStorage<SoFactorsExtractor> extractors;
    private final int limit;
    private final String statsPrefix;
    private final String chartsCategory;
    private final String chartsTitle;
    private final String queueSizeSignal;
    private final String availableSignal;
    private final String limitSignal;
    private volatile boolean closed = false;

    public QueuedNarrowExtractor(
        final List<SoFactorType<?>> inputs,
        final List<SoFactorType<?>> outputs,
        final ObjectsStorage<SoFactorsExtractor> extractors,
        final int limit,
        final String statsPrefix,
        final String chartsCategory,
        final String chartsTitle)
    {
        this.inputs = inputs;
        this.outputs = outputs;
        this.extractors = extractors;
        this.limit = limit;
        this.statsPrefix = statsPrefix;
        this.chartsCategory = chartsCategory;
        this.chartsTitle = chartsTitle;
        queueSizeSignal = statsPrefix + "-queue-size_ammm";
        availableSignal = statsPrefix + "-extractors-available_ammm";
        limitSignal = statsPrefix + "-extractors-limit_ammm";
    }

    @Override
    public void close() {
        closed = true;
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return inputs;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return outputs;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        // Do not perform tasks.add(task); task = tasks.poll();
        // Because this way we have higher chance to execute task which will be
        // cancelled mid air
        execute(null, new Task(context, inputs, callback));
    }

    private void execute(SoFactorsExtractor extractor, Task task) {
        while (!closed) {
            if (task == null) {
                if (extractor == null) {
                    break;
                } else {
                    extractors.put(extractor);
                    extractor = null;
                    task = tasks.poll();
                }
            } else if (task.context.cancelled()) {
                task.callback.cancelled();
                task = tasks.poll();
            } else if (extractor == null) {
                extractor = extractors.get();
                if (extractor == null) {
                    tasks.add(task);
                    task = null;
                    extractor = extractors.get();
                }
            } else {
                extractor.extract(
                    task.context,
                    task.inputs,
                    new IdempotentFutureCallback<>(
                        new Callback(task.callback, extractor)));
                extractor = null;
                task = tasks.poll();
            }
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        if (statsPrefix != null) {
            registry.statersRegistrar().registerStater(this);
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        statsConsumer.stat(queueSizeSignal, tasks.size());
        statsConsumer.stat(availableSignal, extractors.size());
        statsConsumer.stat(limitSignal, limit);
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix)
    {
        String chartsPrefix = statsPrefix + this.statsPrefix;
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChart chart = new GolovanChart(
            "-queue-size",
            " queue size",
            false,
            false,
            0d);
        chart.addSplitSignal(
            config,
            "div(" + statsPrefix + this.queueSizeSignal
            + ',' + statsPrefix + "instance-alive_ammm)",
            1,
            true,
            false);
        group.addChart(chart);

        chart = new GolovanChart(
            "-extractors-available",
            " available extractors",
            false,
            false,
            0d);
        chart.addSignal(
            new GolovanSignal(
                statsPrefix + availableSignal,
                config.tag(),
                "available",
                null,
                1,
                false));
        chart.addSignal(
            new GolovanSignal(
                statsPrefix + limitSignal,
                config.tag(),
                "limit",
                "#ff0000",
                1,
                false));
        group.addChart(chart);

        panel.addCharts(chartsCategory, chartsTitle, group);
    }

    private static class Task {
        private final SoFactorsExtractorContext context;
        private final SoFunctionInputs inputs;
        private final FutureCallback<? super List<SoFactor<?>>> callback;

        Task(
            final SoFactorsExtractorContext context,
            final SoFunctionInputs inputs,
            final FutureCallback<? super List<SoFactor<?>>> callback)
        {
            this.context = context;
            this.inputs = inputs;
            this.callback = callback;
        }
    }

    private class Callback extends FilterFutureCallback<List<SoFactor<?>>> {
        private final SoFactorsExtractor extractor;

        Callback(
            final FutureCallback<? super List<SoFactor<?>>> callback,
            final SoFactorsExtractor extractor)
        {
            super(callback);
            this.extractor = extractor;
        }

        @Override
        public void cancelled() {
            super.cancelled();
            execute(extractor, tasks.poll());
        }

        @Override
        public void completed(final List<SoFactor<?>> result) {
            super.completed(result);
            execute(extractor, tasks.poll());
        }

        @Override
        public void failed(final Exception e) {
            super.failed(e);
            execute(extractor, tasks.poll());
        }
    }
}

