package ru.yandex.mail.so.factors.extractors;

import java.net.InetAddress;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.InetAddressesSoFactorType;
import ru.yandex.mail.so.factors.types.MailMetaSoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.mail.received.ReceivedChainParser;
import ru.yandex.search.document.mail.MailMetaInfo;

public enum SenderIpsExtractor implements SoFactorsExtractor {
    INSTANCE;

    private static final List<SoFactorType<?>> INPUTS = Arrays.asList(
        MailMetaSoFactorType.MAIL_META,
        SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(InetAddressesSoFactorType.INET_ADDRESSES);

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        MailMetaInfo meta = inputs.get(0, MailMetaSoFactorType.MAIL_META);
        InetAddress[] senderIps = extractFromMeta(meta);
        if (senderIps == null) {
            context.logger().info(
                "No ips found in Received, fallback to CONNECT");
            SmtpEnvelopeHolder envelope =
                inputs.get(1, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
            senderIps = extractFromSmtpEnvelope(envelope);
        }
        if (senderIps == null) {
            callback.completed(NULL_RESULT);
        } else {
            callback.completed(
                Collections.singletonList(
                    InetAddressesSoFactorType.INET_ADDRESSES.createFactor(
                        senderIps)));
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        registry.typesRegistry().registerSoFactorType(
            MailMetaSoFactorType.MAIL_META);
        registry.typesRegistry().registerSoFactorType(
            SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
        registry.typesRegistry().registerSoFactorType(
            InetAddressesSoFactorType.INET_ADDRESSES);
    }

    private static InetAddress[] extractFromMeta(final MailMetaInfo meta) {
        if (meta != null) {
            ReceivedChainParser receivedChainParser =
                meta.receivedChainParser();
            if (receivedChainParser != null) {
                List<InetAddress> ips = receivedChainParser.sourceIps();
                int size = ips.size();
                if (size != 0) {
                    return ips.toArray(new InetAddress[size]);
                }
            }
        }
        return null;
    }

    private static InetAddress[] extractFromSmtpEnvelope(
        final SmtpEnvelopeHolder envelope)
    {
        if (envelope != null) {
            InetAddress ip = envelope.ip();
            if (ip != null) {
                return new InetAddress[]{ip};
            }
        }
        return null;
    }
}

