package ru.yandex.mail.so.factors.extractors;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.function.Function;

import core.org.luaj.vm2.Globals;
import core.org.luaj.vm2.LuaClosure;
import core.org.luaj.vm2.Prototype;
import jse.org.luaj.vm2.lib.jse.JsePlatform;

import ru.yandex.http.util.nio.client.AsyncClientRegistrar;
import ru.yandex.http.util.server.ExternalDataProvider;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.so.factors.dsl.DslParser;
import ru.yandex.mail.so.factors.samples.SamplesNotifier;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class SoFactorsExtractorFactoryContext {
    private final Path path;
    private final SoFactorsExtractorsRegistry registry;
    private final Function<String, Consumer<ExtractorStat>>
        statsConsumerFactory;
    private final LongAdder violationsCounter;
    private final LongAdder luaErrorsCounter;
    private final ThreadGroup threadGroup;
    private final AsyncClientRegistrar asyncClientRegistrar;
    private final ExternalDataProvider externalDataProvider;
    private final Map<Path, SoFactorsExtractorsRegistry> modulesCache;
    private final Map<Path, Globals> luaModulesCache;
    private final PrefixedLogger logger;
    private final SamplesNotifier samplesNotifier;
    private final long metricsTimeFrame;

    public SoFactorsExtractorFactoryContext(
        final Path path,
        final SoFactorsExtractorsRegistry registry,
        final Function<String, Consumer<ExtractorStat>> statsConsumerFactory,
        final LongAdder violationsCounter,
        final LongAdder luaErrorsCounter,
        final ThreadGroup threadGroup,
        final AsyncClientRegistrar asyncClientRegistrar,
        final ExternalDataProvider externalDataProvider,
        final Map<Path, SoFactorsExtractorsRegistry> modulesCache,
        final Map<Path, Globals> luaModulesCache,
        final PrefixedLogger logger,
        final SamplesNotifier samplesNotifier,
        final long metricsTimeFrame)
    {
        this.path = path;
        this.registry = registry;
        this.statsConsumerFactory = statsConsumerFactory;
        this.violationsCounter = violationsCounter;
        this.luaErrorsCounter = luaErrorsCounter;
        this.threadGroup = threadGroup;
        this.asyncClientRegistrar = asyncClientRegistrar;
        this.externalDataProvider = externalDataProvider;
        this.modulesCache = modulesCache;
        this.luaModulesCache = luaModulesCache;
        this.logger = logger;
        this.samplesNotifier = samplesNotifier;
        this.metricsTimeFrame = metricsTimeFrame;
    }

    public Path path() {
        return path;
    }

    public SoFactorsExtractorsRegistry registry() {
        return registry;
    }

    public Function<String, Consumer<ExtractorStat>> statsConsumerFactory() {
        return statsConsumerFactory;
    }

    public LongAdder violationsCounter() {
        return violationsCounter;
    }

    public LongAdder luaErrorsCounter() {
        return luaErrorsCounter;
    }

    public ThreadGroup threadGroup() {
        return threadGroup;
    }

    public AsyncClientRegistrar asyncClientRegistrar() {
        return asyncClientRegistrar;
    }

    public ExternalDataProvider externalDataProvider() {
        return externalDataProvider;
    }

    public Map<Path, SoFactorsExtractorsRegistry> modulesCache() {
        return modulesCache;
    }

    public Map<Path, Globals> luaModulesCache() {
        return luaModulesCache;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public SamplesNotifier samplesNotifier() {
        return samplesNotifier;
    }

    public long metricsTimeFrame() {
        return metricsTimeFrame;
    }

    public SoFactorsExtractorsRegistry loadModule(final Path path)
        throws ConfigException, IOException
    {
        SoFactorsExtractorsRegistry localRegistry = modulesCache.get(path);
        if (localRegistry == null) {
            localRegistry = new SoFactorsExtractorsRegistry(registry);
            new DslParser(
                new SoFactorsExtractorFactoryContext(
                    path,
                    localRegistry,
                    statsConsumerFactory,
                    violationsCounter,
                    luaErrorsCounter,
                    threadGroup,
                    asyncClientRegistrar,
                    externalDataProvider,
                    modulesCache,
                    luaModulesCache,
                    logger.addPrefix(path.getFileName().toString()),
                    samplesNotifier,
                    metricsTimeFrame))
                .parse(Files.newBufferedReader(path));
            modulesCache.put(path, localRegistry);
        }
        return localRegistry;
    }

    public Globals loadLuaModule(final Path path)
        throws ConfigException, IOException
    {
        Globals globals = luaModulesCache.get(path);
        if (globals == null) {
            try {
                globals = JsePlatform.standardGlobals();
                Prototype proto =
                    globals.compilePrototype(
                        Files.newInputStream(path),
                        path.toString());
                new LuaClosure(proto, globals).call();
            } catch (RuntimeException e) {
                throw new ConfigException(
                    "Failed to load lua module " + path,
                    e);
            }
            luaModulesCache.put(path, globals);
        }
        return globals;
    }

    public IniConfig readBodyAsIniConfig(final String body)
        throws ConfigException, IOException
    {
        Reader reader = new StringReader(body);
        if (path == null) {
            return new IniConfig(reader);
        } else {
            return new IniConfig(path, reader);
        }
    }
}

