package ru.yandex.mail.so.factors.extractors;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.function.Function;

import core.org.luaj.vm2.Globals;
import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClientRegistrar;
import ru.yandex.http.util.server.ExternalDataProvider;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.config.ImmutableExtractModuleConfig;
import ru.yandex.mail.so.factors.config.ImmutableExtractModulesConfig;
import ru.yandex.mail.so.factors.dsl.DslParser;
import ru.yandex.mail.so.factors.samples.SamplesNotifier;
import ru.yandex.parser.config.ConfigException;

public class SoFactorsExtractorModules
    implements GenericAutoCloseable<IOException>
{
    private final GenericAutoCloseableChain<IOException> closeChain =
        new GenericAutoCloseableChain<>();
    private final SoFactorsExtractorsRegistry registry;
    private final LongAdder violationsCounter;

    public SoFactorsExtractorModules(
        final ImmutableExtractModulesConfig config,
        final LongAdder violationsCounter,
        final SoFactorsExtractorsRegistry registry,
        final Function<String, Consumer<ExtractorStat>> statsConsumerFactory,
        final LongAdder luaErrorsCounter,
        final ThreadGroup threadGroup,
        final AsyncClientRegistrar asyncClientRegistrar,
        final ExternalDataProvider externalDataProvider,
        final PrefixedLogger logger,
        final SamplesNotifier samplesNotifier,
        final long metricsTimeFrame)
        throws ConfigException
    {
        this.registry = new SoFactorsExtractorsRegistry(registry);
        this.violationsCounter = violationsCounter;
        Map<Path, SoFactorsExtractorsRegistry> modulesCache = new HashMap<>();
        Map<Path, Globals> luaModulesCache = new HashMap<>();
        for (Map.Entry<String, ImmutableExtractModuleConfig> entry
            : config.extractModules().entrySet())
        {
            String name = entry.getKey();
            ImmutableExtractModuleConfig moduleConfig = entry.getValue();
            String entryPoint = moduleConfig.entryPoint();
            File dslScript = moduleConfig.dslScript();
            SoFactorsExtractorsRegistry localRegistry =
                new SoFactorsExtractorsRegistry(registry);
            try {
                new DslParser(
                    new SoFactorsExtractorFactoryContext(
                        dslScript.toPath(),
                        localRegistry,
                        statsConsumerFactory,
                        violationsCounter,
                        luaErrorsCounter,
                        threadGroup,
                        asyncClientRegistrar,
                        externalDataProvider,
                        modulesCache,
                        luaModulesCache,
                        logger.addPrefix(name),
                        samplesNotifier,
                        metricsTimeFrame))
                    .parse(Files.newBufferedReader(dslScript.toPath()));
                SoFactorsExtractor extractor =
                    localRegistry.getExtractor(entryPoint);
                if (extractor == null) {
                    throw new ConfigException(
                        "No extractor <" + entryPoint + "> defined in module <"
                        + name + '>');
                }
                closeChain.add(localRegistry);
                if (extractor.outputs().size() != 1) {
                    throw new ConfigException(
                        "Entry point <" + entryPoint
                        + "> must return exactly one factor, got: "
                        + extractor.outputs());
                }
                this.registry.registerExtractor(name, extractor);
            } catch (ConfigException | IOException e) {
                throw new ConfigException(
                    "Failed to construct extractor module <" + name
                    + "> with entry point <" + entryPoint + "> from file: "
                    + dslScript,
                    e);
            }
        }
    }

    @Override
    public void close() throws IOException {
        closeChain.close();
    }

    public LongAdder violationsCounter() {
        return violationsCounter;
    }

    public void extract(
        final String extractorName,
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
        throws HttpException
    {
        // TODO: check inputs and extractor inputs
        SoFactorsExtractor extractor = registry.getExtractor(extractorName);
        if (extractor == null) {
            throw new BadRequestException(
                "Extractor not found <" + extractorName + '>');
        }
        extractor.extract(context, inputs, callback);
    }

    public void extract(
        final String extractorName,
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final JsonType jsonType,
        final FutureCallback<String> callback)
        throws HttpException
    {
        // TODO: check that extractor output is json
        extract(
            extractorName,
            context,
            inputs,
            new Callback(callback, jsonType));
    }

    private static class Callback
        extends AbstractFilterFutureCallback<List<SoFactor<?>>, String>
    {
        private final JsonType jsonType;

        Callback(
            final FutureCallback<String> callback,
            final JsonType jsonType)
        {
            super(callback);
            this.jsonType = jsonType;
        }

        @Override
        public void completed(final List<SoFactor<?>> result) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = jsonType.create(sbw)) {
                Object value = null;
                if (result.size() == 1) {
                    SoFactor<?> factor = result.get(0);
                    if (factor != null) {
                        value = factor.value();
                    }
                }
                writer.value(value);
            } catch (IOException e) {
                failed(e);
                return;
            }
            callback.completed(sbw.toString());
        }
    }
}

