package ru.yandex.mail.so.factors.extractors;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.mail.so.factors.actions.ActionsProcessorExtractorFactory;
import ru.yandex.mail.so.factors.bigb.BigBExtractorFactory;
import ru.yandex.mail.so.factors.blackbox.BlackboxUserinfoExtractorFactory;
import ru.yandex.mail.so.factors.blackbox.BlackboxUserinfoUidExtractor;
import ru.yandex.mail.so.factors.blackbox.BlackboxUserinfosExtractorFactory;
import ru.yandex.mail.so.factors.dssm.DssmEmbeddingExtractorFactory;
import ru.yandex.mail.so.factors.eml2html.Eml2HtmlExtractorFactory;
import ru.yandex.mail.so.factors.eml2html.HtmlImageEmbeddingExtractorFactory;
import ru.yandex.mail.so.factors.fasttext.FastTextEmbeddingExtractorFactory;
import ru.yandex.mail.so.factors.hnsw.HnswDssmDistanceExtractorFactory;
import ru.yandex.mail.so.factors.hnsw.HnswExtractorFactory;
import ru.yandex.mail.so.factors.html2png.Html2PngExtractorFactory;
import ru.yandex.mail.so.factors.predicates.CheckValuePredicateType;
import ru.yandex.mail.so.factors.predicates.FalsePredicate;
import ru.yandex.mail.so.factors.predicates.IsAnyNullPredicateType;
import ru.yandex.mail.so.factors.predicates.IsFalsePredicate;
import ru.yandex.mail.so.factors.predicates.IsTruePredicate;
import ru.yandex.mail.so.factors.predicates.NonEmptyJsonObjectPredicate;
import ru.yandex.mail.so.factors.predicates.NonEmptyStringPredicate;
import ru.yandex.mail.so.factors.predicates.SimpleSoPredicateType;
import ru.yandex.mail.so.factors.predicates.SoPredicate;
import ru.yandex.mail.so.factors.predicates.SoPredicateType;
import ru.yandex.mail.so.factors.predicates.StringsEqualPredicate;
import ru.yandex.mail.so.factors.predicates.TruePredicate;
import ru.yandex.mail.so.factors.senders.SendersExtractorFactory;
import ru.yandex.mail.so.factors.senders.SendersInfoExtractor;
import ru.yandex.mail.so.factors.sherlock.TemplateMasterExtractorFactory;
import ru.yandex.mail.so.factors.shinglers.activity.ActivityExtractorFactory;
import ru.yandex.mail.so.factors.types.SoFactorTypesRegistry;
import ru.yandex.mail.so.factors.ugc.UgcExtractorFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.StatersRegistrar;

public class SoFactorsExtractorsRegistry
    implements GenericAutoCloseable<IOException>
{
    private final GenericAutoCloseableChain<IOException> chain =
        new GenericAutoCloseableChain<>();
    private final StatersRegistrar statersRegistrar;
    private final SoFactorTypesRegistry typesRegistry;
    private final Map<String, SoFactorsExtractor> extractorsRegistry;
    private final Map<String, SoFactorsExtractorFactory>
        extractorsFactoriesRegistry;
    private final Map<String, SimpleSoFactorsExtractorFactory>
        simpleExtractorsFactoriesRegistry;
    private final Map<String, SoPredicate> predicatesRegistry;
    private final Map<String, SoPredicateType> predicateTypesRegistry;
    private final Map<String, SimpleSoPredicateType>
        simplePredicateTypesRegistry;

    public SoFactorsExtractorsRegistry(
        final StatersRegistrar statersRegistrar,
        final SoFactorTypesRegistry typesRegistry)
        throws ConfigException
    {
        this.statersRegistrar = statersRegistrar;
        this.typesRegistry = new SoFactorTypesRegistry(typesRegistry);
        extractorsRegistry = new HashMap<>();
        registerExtractor(
            "convert_tab_pfs_to_actions",
            ConvertTabPfsToActionsExtractor.INSTANCE);
        registerExtractor(
            "create_json_map",
            CreateJsonMapExtractor.INSTANCE);
        registerExtractor(
            "detect_language",
            LanguageDetectorExtractor.INSTANCE);
        registerExtractor(
            "dkim_domains",
            DkimDomainsExtractor.INSTANCE);
        registerExtractor(
            "filter_tab_moves",
            FilterTabMovesExtractor.INSTANCE);
        registerExtractor(
            "from_domain",
            FromDomainExtractor.INSTANCE);
        registerExtractor(
            "full_body_fnv64",
            FullBodyFnv64Extractor.INSTANCE);
        registerExtractor(
            "hbf_project_id",
            HbfProjectIdExtractor.INSTANCE);
        registerExtractor(
            "header",
            HeaderExtractor.INSTANCE);
        registerExtractor(
            "html_text",
            HtmlTextExtractor.INSTANCE);
        registerExtractor(
            "join_string_list",
            JoinStringListExtractor.INSTANCE);
        registerExtractor(
            "limit_words",
            LimitWordsExtractor.INSTANCE);
        registerExtractor(
            "lowercase_headers",
            LowercaseHeadersExtractor.INSTANCE);
        registerExtractor(
            "mail_fnv64",
            MailFnv64Extractor.INSTANCE);
        registerExtractor(
            "nested_mail",
            NestedMailExtractor.INSTANCE);
        registerExtractor(
            "not",
            NotExtractor.INSTANCE);
        registerExtractor(
            "senders_info",
            SendersInfoExtractor.INSTANCE);
        registerExtractor(
            "sender_ips",
            SenderIpsExtractor.INSTANCE);
        registerExtractor(
            "tab_pf",
            TabPfExtractor.INSTANCE);
        registerExtractor(
            "text_part",
            TextPartExtractor.INSTANCE);
        registerExtractor(
            "trim_string",
            TrimStringExtractor.INSTANCE);
        registerExtractor(
            "urls",
            UrlsExtractor.INSTANCE);
        registerExtractor(
            "userinfo_uid",
            BlackboxUserinfoUidExtractor.INSTANCE);

        extractorsFactoriesRegistry = new HashMap<>();
        registerExtractorFactory(
            "actions_processor",
            ActionsProcessorExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "activity",
            ActivityExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "bigb",
            BigBExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "binary_jniwapper",
            BinaryJniWrapperExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "blackbox_userinfo",
            BlackboxUserinfoExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "blackbox_userinfos",
            BlackboxUserinfosExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "compose_doc",
            ComposeDocExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "concat",
            ConcatExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "dssm",
            DssmEmbeddingExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "dkim_stats",
            DkimStatsExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "embed_html_images",
            HtmlImageEmbeddingExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "eml2html",
            Eml2HtmlExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "fast_text",
            FastTextEmbeddingExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "filter_search",
            FilterSearchExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "for_each_json_object",
            ForEachJsonObjectExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "for_each_tikaite_doc",
            ForEachTikaiteDocExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "hnsw",
            HnswExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "hnsw_dssm",
            HnswDssmDistanceExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "html2png",
            Html2PngExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "http_get",
            HttpGetExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "jniwrapper",
            JniWrapperExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "join_tikaite_docs_field",
            JoinTikaiteDocsFieldExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "log",
            LogExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "narrow",
            NarrowExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "org_settings",
            OrgSettingsExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "received_chain",
            ReceivedChainExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "sed",
            SedExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "senders",
            SendersExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "template_master",
            TemplateMasterExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "thread_pool",
            ThreadPoolExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "tikaite",
            TikaiteExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "ugc",
            UgcExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "unperson",
            UnpersonExtractorFactory.INSTANCE);
        registerExtractorFactory(
            "yadisk_info",
            YaDiskInfoExtractorFactory.INSTANCE);

        simpleExtractorsFactoriesRegistry = new HashMap<>();
        registerSimpleExtractorFactory(
            "assert_not_null",
            AssertNotNullExtractorFactory.INSTANCE);
        registerSimpleExtractorFactory(
            "identity",
            IdentityExtractorFactory.INSTANCE);
        registerSimpleExtractorFactory(
            "is_any_null",
            IsAnyNullExtractorFactory.INSTANCE);

        predicatesRegistry = new HashMap<>();
        registerPredicate(
            "false",
            FalsePredicate.INSTANCE);
        registerPredicate(
            "is_false",
            IsFalsePredicate.INSTANCE);
        registerPredicate(
            "is_true",
            IsTruePredicate.INSTANCE);
        registerPredicate(
            "non_empty_json_object",
            NonEmptyJsonObjectPredicate.INSTANCE);
        registerPredicate(
            "non_empty_string",
            NonEmptyStringPredicate.INSTANCE);
        registerPredicate(
            "strings_equal",
            StringsEqualPredicate.INSTANCE);
        registerPredicate(
            "true",
            TruePredicate.INSTANCE);

        predicateTypesRegistry = new HashMap<>();
        registerPredicateType(
            "check_value",
            CheckValuePredicateType.INSTANCE);

        simplePredicateTypesRegistry = new HashMap<>();
        registerSimplePredicateType(
            "is_any_null",
            IsAnyNullPredicateType.INSTANCE);
    }

    // Doesn't take ownership of close chain from original extractorsRegistry
    public SoFactorsExtractorsRegistry(
        final SoFactorsExtractorsRegistry registry)
    {
        statersRegistrar = registry.statersRegistrar;
        typesRegistry = new SoFactorTypesRegistry(registry.typesRegistry);
        extractorsRegistry = new HashMap<>(registry.extractorsRegistry);
        extractorsFactoriesRegistry =
            new HashMap<>(registry.extractorsFactoriesRegistry);
        simpleExtractorsFactoriesRegistry =
            new HashMap<>(registry.simpleExtractorsFactoriesRegistry);
        predicateTypesRegistry =
            new HashMap<>(registry.predicateTypesRegistry);
        predicatesRegistry = new HashMap<>(registry.predicatesRegistry);
        simplePredicateTypesRegistry =
            new HashMap<>(registry.simplePredicateTypesRegistry);
    }

    public StatersRegistrar statersRegistrar() {
        return statersRegistrar;
    }

    public SoFactorTypesRegistry typesRegistry() {
        return typesRegistry;
    }

    public void addNestedRegistry(final SoFactorsExtractorsRegistry registry) {
        chain.add(registry);
    }

    public void addAnonymousExtractor(final SoFactorsExtractor extractor)
        throws ConfigException
    {
        chain.add(extractor);
        extractor.registerInternals(this);
    }

    public void addAnonymousPredicate(final SoPredicate predicate)
        throws ConfigException
    {
        chain.add(predicate);
        predicate.registerInternals(this);
    }

    public void registerExtractor(
        final String name,
        final SoFactorsExtractor extractor)
        throws ConfigException
    {
        chain.add(extractor);
        SoFactorsExtractor oldExtractor =
            extractorsRegistry.putIfAbsent(name, extractor);
        if (oldExtractor != null && extractor != oldExtractor) {
            throw new ConfigException(
                "Can't add so factor extractor " + extractor + " for name "
                + name + " as there is already extractor " + oldExtractor
                + " present for this name");
        }
        try {
            extractor.registerInternals(this);
        } catch (ConfigException e) {
            throw new ConfigException(
                "Failed to register internals for <" + name + '>',
                e);
        }
    }

    public SoFactorsExtractor getExtractor(final String name) {
        return extractorsRegistry.get(name);
    }

    public void registerExtractorFactory(
        final String name,
        final SoFactorsExtractorFactory factory)
        throws ConfigException
    {
        chain.add(factory);
        SoFactorsExtractorFactory oldFactory =
            extractorsFactoriesRegistry.putIfAbsent(name, factory);
        if (oldFactory != null) {
            throw new ConfigException(
                "Can't add so factor extractor factory " + factory
                + " for name " + name + " as there is already factory "
                + oldFactory + " present for this name");
        }
        try {
            factory.registerInternals(this);
        } catch (ConfigException e) {
            throw new ConfigException(
                "Failed to register internals for <" + name + '>',
                e);
        }
    }

    public SoFactorsExtractorFactory getExtractorFactory(final String name) {
        return extractorsFactoriesRegistry.get(name);
    }

    public void registerSimpleExtractorFactory(
        final String name,
        final SimpleSoFactorsExtractorFactory factory)
        throws ConfigException
    {
        chain.add(factory);
        SimpleSoFactorsExtractorFactory oldFactory =
            simpleExtractorsFactoriesRegistry.putIfAbsent(name, factory);
        if (oldFactory != null) {
            throw new ConfigException(
                "Can't add simple so factor extractor factory " + factory
                + " for name " + name + " as there is already factory "
                + oldFactory + " present for this name");
        }
        try {
            factory.registerInternals(this);
        } catch (ConfigException e) {
            throw new ConfigException(
                "Failed to register internals for <" + name + '>',
                e);
        }
    }

    public SimpleSoFactorsExtractorFactory getSimpleExtractorFactory(
        final String name)
    {
        return simpleExtractorsFactoriesRegistry.get(name);
    }

    public void registerPredicate(
        final String name,
        final SoPredicate predicate)
        throws ConfigException
    {
        chain.add(predicate);
        SoPredicate oldPredicate =
            predicatesRegistry.putIfAbsent(name, predicate);
        if (oldPredicate != null) {
            throw new ConfigException(
                "Can't add so predicate " + predicate
                + " for name " + name + " as there is already predicate "
                + oldPredicate + " present for this name");
        }
        try {
            predicate.registerInternals(this);
        } catch (ConfigException e) {
            throw new ConfigException(
                "Failed to register internals for <" + name + '>',
                e);
        }
    }

    public SoPredicate getPredicate(final String name) {
        return predicatesRegistry.get(name);
    }

    public void registerPredicateType(
        final String name,
        final SoPredicateType predicateType)
        throws ConfigException
    {
        chain.add(predicateType);
        SoPredicateType oldType =
            predicateTypesRegistry.putIfAbsent(name, predicateType);
        if (oldType != null) {
            throw new ConfigException(
                "Can't add so predicate type " + predicateType
                + " for name " + name + " as there is already predicate type "
                + oldType + " present for this name");
        }
        try {
            predicateType.registerInternals(this);
        } catch (ConfigException e) {
            throw new ConfigException(
                "Failed to register internals for <" + name + '>',
                e);
        }
    }

    public SoPredicateType getPredicateType(final String name) {
        return predicateTypesRegistry.get(name);
    }

    public void registerSimplePredicateType(
        final String name,
        final SimpleSoPredicateType predicateType)
        throws ConfigException
    {
        chain.add(predicateType);
        SimpleSoPredicateType oldType =
            simplePredicateTypesRegistry.putIfAbsent(name, predicateType);
        if (oldType != null) {
            throw new ConfigException(
                "Can't add simple so predicate type " + predicateType
                + " for name " + name + " as there is already predicate type "
                + oldType + " present for this name");
        }
        try {
            predicateType.registerInternals(this);
        } catch (ConfigException e) {
            throw new ConfigException(
                "Failed to register internals for <" + name + '>',
                e);
        }
    }

    public SimpleSoPredicateType getSimplePredicateType(final String name) {
        return simplePredicateTypesRegistry.get(name);
    }

    @Override
    public void close() throws IOException {
        chain.close();
    }
}

