package ru.yandex.mail.so.factors.extractors;

import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.LoggingRejectedExecutionHandler;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.stater.ThreadPoolStater;

public class ThreadPoolExtractor implements SoFactorsExtractor {
    private final SoFactorsExtractor extractor;
    private final ThreadPoolExecutor threadPool;

    public ThreadPoolExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        int workers = config.get("workers", PositiveIntegerValidator.INSTANCE);
        int queueSize =
            config.get("queue-size", PositiveIntegerValidator.INSTANCE);
        String extractorName = config.getString("extractor");
        extractor = context.registry().getExtractor(extractorName);
        if (extractor == null) {
            throw new ConfigException(
                "Extractor <" + extractorName + "> not found");
        }

        LongAdder rejectedTasksCounter = new LongAdder();
        threadPool = new ThreadPoolExecutor(
            workers,
            workers,
            1,
            TimeUnit.MINUTES,
            new LifoWaitBlockingQueue<>(queueSize),
            new NamedThreadFactory(context.threadGroup(), name + '-', true),
            new LoggingRejectedExecutionHandler(
                context.logger().addPrefix(name),
                rejectedTasksCounter));
        threadPool.prestartAllCoreThreads();
        context.registry().statersRegistrar().registerStater(
            new ThreadPoolStater(
                threadPool,
                rejectedTasksCounter,
                name + '-'));
    }

    @Override
    public void close() {
        threadPool.shutdown();
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return extractor.inputs();
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return extractor.outputs();
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        SeparateThreadPoolSoFactorsExtractorContext separateThreadPoolContext =
            new SeparateThreadPoolSoFactorsExtractorContext(
                context,
                threadPool);
        try {
            threadPool.execute(
                new Task(
                    extractor,
                    separateThreadPoolContext,
                    inputs,
                    callback));
        } catch (RuntimeException e) {
            callback.failed(e);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry) {
    }

    private static class SeparateThreadPoolSoFactorsExtractorContext
        extends FilterSoFactorsExtractorContext
    {
        private final ThreadPoolExecutor threadPool;

        SeparateThreadPoolSoFactorsExtractorContext(
            final SoFactorsExtractorContext context,
            final ThreadPoolExecutor threadPool)
        {
            super(context);
            this.threadPool = threadPool;
        }

        @Override
        public ThreadPoolExecutor executor() {
            return threadPool;
        }
    }

    private static class Task implements Runnable {
        private final SoFactorsExtractor extractor;
        private final SoFactorsExtractorContext context;
        private final SoFunctionInputs inputs;
        private final FutureCallback<? super List<SoFactor<?>>> callback;

        Task(
            final SoFactorsExtractor extractor,
            final SoFactorsExtractorContext context,
            final SoFunctionInputs inputs,
            final FutureCallback<? super List<SoFactor<?>>> callback)
        {
            this.extractor = extractor;
            this.context = context;
            this.inputs = inputs;
            this.callback = callback;
        }

        @Override
        public void run() {
            try {
                extractor.extract(context, inputs, callback);
            } catch (Exception e) {
                callback.failed(e);
            } catch (Error e) {
                callback.failed(new Exception(e));
            }
        }
    }
}

