package ru.yandex.mail.so.factors.extractors;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.CollectionConsumer;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.io.TrimmingWriter;
import ru.yandex.jniwrapper.ImmutableJniWrapperConfig;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperConfigBuilder;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.url.processor.ImmutableUrlProcessorConfig;
import ru.yandex.url.processor.UrlInfo;
import ru.yandex.url.processor.UrlProcessor;
import ru.yandex.url.processor.UrlProcessorConfigBuilder;

public class UnpersonExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(StringSoFactorType.STRING);
    private static final char[] URL_PLACEHOLDER = " %Uri% ".toCharArray();

    private final ImmutableUrlProcessorConfig urlProcessorConfig;
    private final JniWrapper jniWrapper;
    private final long maxTokens;

    public UnpersonExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        this(
            new UrlProcessorConfigBuilder(config).build(),
            createJniWrapper(name, context, config),
            config.get("max-tokens", PositiveLongValidator.INSTANCE));
    }

    public UnpersonExtractor(
        final ImmutableUrlProcessorConfig urlProcessorConfig,
        final JniWrapper jniWrapper,
        final long maxTokens)
    {
        this.urlProcessorConfig = urlProcessorConfig;
        this.jniWrapper = jniWrapper;
        this.maxTokens = maxTokens;
    }

    private static JniWrapper createJniWrapper(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableJniWrapperConfig jniwrapperConfig =
            new JniWrapperConfigBuilder(config).build();

        ThreadGroup threadGroup = context.threadGroup();
        try {
            return JniWrapper.create(
                jniwrapperConfig,
                new ThreadFactoryConfig(
                    threadGroup.getName() + '-' + name + '-')
                    .group(threadGroup)
                    .daemon(true));
        } catch (JniWrapperException e) {
            throw new ConfigException("Failed to construct JniWrapper", e);
        }
    }

    @Override
    public void close() {
        jniWrapper.close();
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return INPUTS;
    }

    private static Match findAny(
        final String haystack,
        final int startPos,
        final String needle1,
        final String needle2)
    {
        int idx1 = haystack.indexOf(needle1, startPos);
        int idx2 = haystack.indexOf(needle2, startPos);
        if (idx1 == -1) {
            if (idx2 == -1) {
                return null;
            } else {
                return new Match(idx2, idx2 + needle2.length());
            }
        } else if (idx2 == -1 || idx2 > idx1) {
            return new Match(idx1, idx1 + needle1.length());
        } else {
            return new Match(idx2, idx2 + needle2.length());
        }
    }

    private static int findSuffix(
        final String str,
        final int pos,
        final String suffix)
    {
        int strLen = str.length();
        int suffixLen = suffix.length();
        int tailLen = strLen - pos;
        if (suffixLen > tailLen) {
            return -1;
        }
        int end = pos + suffixLen;
        if (str.substring(pos, end).equals(suffix)) {
            return end;
        }
        if (suffixLen > tailLen - 1) {
            return -1;
        }
        ++end;
        if (str.substring(pos + 1, end).equals(suffix)) {
            return end;
        }
        return -1;
    }

    private String unpersonText(final String str)
        throws IOException, JniWrapperException
    {
        List<UrlInfo> urlInfos = new ArrayList<>();
        UrlProcessor processor = new UrlProcessor(
            new CollectionConsumer<>(urlInfos),
            urlProcessorConfig);
        int len = str.length();
        processor.process(str.toCharArray(), 0, len);
        processor.process();

        StringBuilder sb = new StringBuilder(len);
        StringBuilderWriter sbw = new StringBuilderWriter(sb);
        int size = urlInfos.size();
        int prev = 0;
        long tokensCount = 0;
        char[] buf = new char[Math.min(1024, len)];
        try (Writer writer = new TrimmingWriter(sbw)) {
            for (int i = 0; i < size && tokensCount < maxTokens; ++i) {
                UrlInfo urlInfo = urlInfos.get(i);
                int start = urlInfo.start();
                if (start > prev) {
                    String part =
                        jniWrapper.apply(str.substring(prev, start), null)
                            .process(null, 0, 0);
                    int partLen = part.length();
                    if (partLen > buf.length) {
                        buf = new char[Math.max(partLen, buf.length << 1)];
                    }
                    part.getChars(0, partLen, buf, 0);
                    writer.write(buf, 0, partLen);

                    // Count number of tokens in deobfuscated part
                    boolean inWord = false;
                    for (int j = 0; j < partLen; ++j) {
                        if (Character.isLetterOrDigit(buf[j])) {
                            inWord = true;
                        } else if (inWord) {
                            inWord = false;
                            ++tokensCount;
                        }
                    }
                    if (inWord) {
                        ++tokensCount;
                    }
                }
                writer.write(URL_PLACEHOLDER);
                ++tokensCount;
                prev = urlInfo.end();
            }
            if (prev < len && tokensCount < maxTokens) {
                writer.write(
                    jniWrapper.apply(str.substring(prev, len), null)
                        .process(null, 0, 0));
            }
        }
        if (sb.length() == 0) {
            return null;
        } else {
            String result = sb.toString();
            sb.setLength(0);
            int pos = 0;
            while (true) {
                Match matchPassword =
                    findAny(result, pos, "%ShortPassword%", "%Password%");
                Match matchNumber =
                    findAny(result, pos, "%ShortNumber%", "%Number%");
                boolean hasPassword = matchPassword != null;
                Match match;
                if (matchPassword == null) {
                    match = matchNumber;
                } else if (matchNumber == null
                    || matchPassword.start < matchNumber.start)
                {
                    match = matchPassword;
                } else {
                    match = matchNumber;
                }
                if (match == null) {
                    if (sb.length() > 0) {
                        sb.append(result, pos, result.length());
                        result = sb.toString();
                    }
                    break;
                }
                int end = match.end;
                while ((end < result.length() && result.charAt(end) == '%')
                    || (end + 1 < result.length()
                        && result.charAt(end + 1) == '%'))
                {
                    int idx = findSuffix(result, end, "%Number%");
                    if (idx == -1) {
                        idx = findSuffix(result, end, "%Password%");
                        if (idx == -1) {
                            idx = findSuffix(result, end, "%ShortPassword%");
                            if (idx == -1) {
                                idx = findSuffix(result, end, "%ShortNumber%");
                            } else {
                                hasPassword = true;
                            }
                        } else {
                            hasPassword = true;
                        }
                    }
                    if (idx == -1) {
                        break;
                    } else {
                        end = idx;
                    }
                }
                sb.append(result, pos, match.start);
                if (end != match.end) {
                    if (hasPassword) {
                        sb.append("%Password%");
                    } else {
                        sb.append("%Number%");
                    }
                } else {
                    sb.append(result, match.start, match.end);
                }
                pos = end;
            }
            return result;
        }
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        String str = inputs.get(0, StringSoFactorType.STRING);
        if (str != null && !str.isEmpty()) {
            try {
                String result = unpersonText(str);
                if (result != null) {
                    callback.completed(
                        Collections.singletonList(
                            StringSoFactorType.STRING.createFactor(result)));
                    return;
                }
            } catch (IOException | JniWrapperException e) {
                callback.failed(e);
                return;
            }
        }

        callback.completed(NULL_RESULT);
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        UnpersonExtractorFactory.INSTANCE.registerInternals(registry);
    }

    public static void main(final String... args)
        throws ConfigException, IOException, JniWrapperException
    {
        if (args.length != 1) {
            System.err.println(
                "Usage: " + UnpersonExtractor.class.getCanonicalName()
                + " <config-file>");
            System.exit(1);
        }
        IniConfig config = new IniConfig(Paths.get(args[0]));

        UnpersonExtractor extractor = new UnpersonExtractor(
            new UrlProcessorConfigBuilder(config).build(),
            JniWrapper.create(
                new JniWrapperConfigBuilder(config).build(),
                new ThreadFactoryConfig("UnpersonExtractor-").daemon(true)),
            Long.MAX_VALUE);

        config.checkUnusedKeys();

        try (BufferedReader reader =
                new BufferedReader(
                    new InputStreamReader(System.in, StandardCharsets.UTF_8)))
        {
            while (true) {
                String line = reader.readLine();
                if (line == null) {
                    break;
                }
                line = extractor.unpersonText(line);
                if (line != null) {
                    System.out.println(line);
                }
            }
        }
    }

    private static class Match {
        private final int start;
        private final int end;

        Match(final int start, final int end) {
            this.start = start;
            this.end = end;
        }
    }
}

