package ru.yandex.mail.so.factors.extractors;

import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.regex.Pattern;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.LoggingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocsSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.search.document.mail.MailMetaInfo;

public class YaDiskInfoExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(TikaiteDocsSoFactorType.TIKAITE_DOCS);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(JsonObjectSoFactorType.JSON_OBJECT);

    private static final Pattern DISK_URL_PATTERN =
        Pattern.compile(
            "https?://"
            + "(yadi[.]sk|disk[.]yandex[.](?:ru|com|ua|kz|tr|by|net|com[.]tr))"
            + "/[a-z]+/.*");

    private final AsyncClient client;
    private final HttpHost host;

    public YaDiskInfoExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableHttpHostConfig hostConfig =
            new HttpHostConfigBuilder(config).build();
        client = context.asyncClientRegistrar().client(name, hostConfig);
        host = hostConfig.host();
    }

    @Override
    public void close() {
        // Client already handled by registrar
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        JsonList docs = inputs.get(0, TikaiteDocsSoFactorType.TIKAITE_DOCS);
        if (docs == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        MultiFutureCallback<String> multiCallback =
            new MultiFutureCallback<>(new Callback(callback));
        AsyncClient client = this.client.adjust(context.httpContext());
        Supplier<? extends HttpClientContext> contextGenerator =
            context.requestsListener().createContextGeneratorFor(client);
        int size = docs.size();
        int count = 0;
        try {
            for (int i = 0; i < size; ++i) {
                JsonObject doc = docs.get(i);
                String urls = doc.get(MailMetaInfo.X_URLS).asStringOrNull();
                if (urls != null) {
                    for (String url: urls.split("\n")) {
                        if (DISK_URL_PATTERN.matcher(url).matches()) {
                            PrefixedLogger logger =
                                context.logger().addPrefix(
                                    Integer.toString(count));
                            if (logger.isLoggable(Level.FINE)) {
                                logger.fine(
                                    "Requesting yadisk url info: " + url);
                            }
                            QueryConstructor query =
                                new QueryConstructor(
                                    "/v1/disk/public/resources?limit=1");
                            ScanningCgiParams params =
                                new ScanningCgiParams(
                                    new UriParser(url).queryParser());
                            String hash = params.getString("hash", "");
                            if (hash.length() > 0) {
                                query.append("public_key", hash);
                            } else {
                                query.append("public_key", url);
                            }
                            BasicAsyncRequestProducerGenerator
                                producerGenerator =
                                    new BasicAsyncRequestProducerGenerator(
                                        query.toString());
                            producerGenerator.addHeader("X-Uid", "0");
                            client.execute(
                                host,
                                producerGenerator,
                                AsyncStringConsumerFactory.OK,
                                contextGenerator,
                                new LoggingFutureCallback<>(
                                    new ErrorSuppressingFutureCallback<>(
                                        multiCallback.newCallback(),
                                        ""),
                                    logger));
                            if (++count >= 5) {
                                multiCallback.done();
                                return;
                            }
                        }
                    }
                }
            }
            multiCallback.done();
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        YaDiskInfoExtractorFactory.INSTANCE.registerInternals(registry);
    }

    private static class Callback
        extends AbstractFilterFutureCallback<List<String>, List<SoFactor<?>>>
    {
        Callback(final FutureCallback<? super List<SoFactor<?>>> callback) {
            super(callback);
        }

        @Override
        public void completed(final List<String> result) {
            int size = result.size();
            JsonList list = new JsonList(BasicContainerFactory.INSTANCE, size);
            for (int i = 0; i < size; ++i) {
                String info = result.get(i);
                if (info.length() > 0) {
                    list.add(new JsonString(info));
                }
            }
            if (list.isEmpty()) {
                callback.completed(NULL_RESULT);
            } else {
                callback.completed(
                    Collections.singletonList(
                        JsonObjectSoFactorType.JSON_OBJECT.createFactor(list)));
            }
        }
    }
}

