package ru.yandex.mail.so.factors.html2png;

import java.io.File;
import java.io.IOException;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;
import org.openqa.selenium.Dimension;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxDriverLogLevel;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.firefox.GeckoDriverService;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.charset.Encoder;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.StringSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class Html2PngExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Collections.singletonList(StringSoFactorType.STRING);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(BinarySoFactorType.BINARY);

    private static final String PREFIX =
        "data:text/html;charset=utf-8;base64,";
    private static final int PREFIX_LENGTH = PREFIX.length();

    private final FirefoxDriver driver;

    public Html2PngExtractor(final IniConfig config) throws ConfigException {
        String driverPath = config.getString("driver-path");
        String binaryPath = config.getString("binary-path");
        int width = config.getInt("width");
        int height = config.getInt("height");
        driver =
            new FirefoxDriver(
                new GeckoDriverService.Builder()
                    .usingDriverExecutable(new File(driverPath))
                    .withLogFile(new File("/dev/null"))
                    .build(),
                new FirefoxOptions()
                    .setHeadless(true)
                    .setBinary(binaryPath)
                    .setLogLevel(FirefoxDriverLogLevel.FATAL));
        driver.manage().window().setSize(new Dimension(width, height));
    }

    @Override
    public void close() throws IOException {
        try {
            driver.quit();
        } catch (Throwable t) {
            throw new IOException("Failed to close webdriver", t);
        }
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        String html = inputs.get(0, StringSoFactorType.STRING);
        if (html == null || html.isEmpty()) {
            context.logger().info("No HTML to render");
            callback.completed(NULL_RESULT);
            return;
        }
        StringBuilder sb = new StringBuilder(
            PREFIX_LENGTH + (html.length() + 2) / 3 * 4);
        sb.append(PREFIX);
        try {
            Encoder encoder = new Encoder(
                StandardCharsets.UTF_8.newEncoder()
                    .onMalformedInput(CodingErrorAction.REPLACE)
                    .onUnmappableCharacter(CodingErrorAction.REPLACE));
            encoder.process(html.toCharArray());
            Base64Encoder base64Encoder = new Base64Encoder();
            encoder.processWith(base64Encoder);
            encoder = null;
            base64Encoder.processWith(new StringBuilderProcessorAdapter(sb));
        } catch (CharacterCodingException e) {
            callback.failed(e);
            return;
        }
        String text = new String(sb);
        sb = null;
        try {
            driver.get(text);
            text = null;
            byte[] png = driver.getScreenshotAs(OutputType.BYTES);
            callback.completed(
                Collections.singletonList(
                    BinarySoFactorType.BINARY.createFactor(
                        new ByteArrayProcessable(png))));
        } catch (RuntimeException e) {
            callback.failed(e);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        Html2PngExtractorFactory.INSTANCE.registerInternals(registry);
    }
}

