package ru.yandex.mail.so.factors.samples;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.util.string.StringUtils;

public class SamplesLoader implements SamplesNotifier {
    private final Queue<SamplesSubscriber> subscribers =
        new ConcurrentLinkedQueue<>();

    @Override
    public void subscribe(final SamplesSubscriber subscriber) {
        subscribers.add(subscriber);
    }

    public void loadFromFile(final File file, final Logger logger)
        throws IOException
    {
        try (BufferedReader reader =
                new BufferedReader(
                    new FileReader(file, StandardCharsets.UTF_8)))
        {
            loadFromReader(reader, logger);
        }
    }

    public void loadFromString(final String str, final Logger logger) {
        try (BufferedReader reader =
                new BufferedReader(new StringReader(str)))
        {
            loadFromReader(reader, logger);
        } catch (IOException e) {
            // Impossible situation
            throw new RuntimeException(e);
        }
    }

    public void loadFromReader(
        final BufferedReader reader,
        final Logger logger)
        throws IOException
    {
        while (true) {
            String line = reader.readLine();
            if (line == null) {
                break;
            }
            try {
                loadLine(line, logger);
            } catch (JsonException e) {
                logger.log(Level.WARNING, "Malformed line <" + line + '>', e);
            }
        }
    }

    private void loadLine(final String line, final Logger logger)
        throws JsonException
    {
        if (line.isEmpty()) {
            return;
        }

        JsonMap root = TypesafeValueContentHandler.parse(line).asMap();
        String id = root.get("url").asString();
        JsonList docs = root.getListOrNull("docs");
        if (docs == null) {
            for (SamplesSubscriber subscriber: subscribers) {
                subscriber.removeSample(id);
            }
        } else {
            String labels =
                StringUtils.nullifyEmpty(root.get("labels").asStringOrNull());
            String fromDomain = null;
            if (!docs.isEmpty()) {
                JsonMap doc = docs.get(0).asMap();
                String from = doc.get("hdr_from_normalized").asStringOrNull();
                if (from != null) {
                    from = StringUtils.firstLine(from).trim();
                    fromDomain = MailAliases.INSTANCE.parseAndNormalize(from)
                        .getValue();
                }
            }
            for (SamplesSubscriber subscriber: subscribers) {
                try {
                    subscriber.addSample(id, labels, fromDomain, docs);
                } catch (SamplesSubscriberException e) {
                    logger.log(
                        Level.WARNING,
                        "Samples subscriber " + subscriber
                        + " failed to load sample <" + line + '>',
                        e);
                }
            }
        }
    }
}

