package ru.yandex.mail.so.factors.senders;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.collection.LongPair;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.mail.senders.SenderInfo;

public class SendersClient extends AbstractAsyncClient<SendersClient> {
    private final HttpHost host;

    public SendersClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpHostConfig config)
    {
        super(reactor, config);
        host = config.host();
    }

    private SendersClient(
        final CloseableHttpAsyncClient client,
        final SendersClient sample)
    {
        super(client, sample);
        host = sample.host;
    }

    @Override
    public SendersClient adjust(final CloseableHttpAsyncClient client) {
        return new SendersClient(client, this);
    }

    public Future<SendersResponse> sendersInfo(
        // uids and suids (possibly null)
        final List<LongPair<Long>> users,
        // senders addrs
        final List<SenderInfo> senders,
        // sender host from Received: headers
        final String senderHost,
        final Long senderUid,
        final List<String> inReplyTo,
        final List<String> references,
        final SendersResponseListener listener,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<SendersResponse> callback)
        throws BadRequestException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
            writer.startObject();
            writer.key("requests");
            writer.startArray();
            for (LongPair<Long> user: users) {
                writer.startObject();
                writer.key("uid");
                writer.value(user.first());
                Long suid = user.second();
                if (suid != null) {
                    writer.key("suid");
                    writer.value(suid);
                }
                for (SenderInfo senderInfo: senders) {
                    String email = senderInfo.email();
                    int sep = MailAliases.emailSeparatorPos(email);
                    String domain;
                    if (sep == -1) {
                        domain = email;
                    } else {
                        domain = email.substring(sep + 1);
                        writer.key(senderInfo.type().emailField());
                        writer.value(email);
                    }
                    writer.key(senderInfo.type().domainField());
                    writer.value(domain);
                }
                writer.key("sender-host");
                writer.value(senderHost);

                int size = inReplyTo.size();
                if (size > 0) {
                    writer.key("in-reply-to");
                    writer.startArray();
                    for (int i = 0; i < size; ++i) {
                        writer.value(inReplyTo.get(i));
                    }
                    writer.endArray();
                }

                size = references.size();
                if (size > 0) {
                    writer.key("references");
                    writer.startArray();
                    for (int i = 0; i < size; ++i) {
                        writer.value(references.get(i));
                    }
                    writer.endArray();
                }
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            throw new BadRequestException(e);
        }
        Callback callbackProxy = new Callback(callback, listener);
        String uri = "/api/async/senders?names-max=1&json-type=dollar";
        if (senderUid != null) {
            uri += "&sender-uid=" + senderUid.longValue();
        }
        execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                uri,
                sbw.toString(),
                ContentType.APPLICATION_JSON.withCharset(requestCharset())),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            contextGenerator,
            callbackProxy);
        return callbackProxy;
    }

    private static class Callback
        extends CallbackFutureBase<SendersResponse, JsonObject>
    {
        private final SendersResponseListener listener;

        Callback(
            final FutureCallback<SendersResponse> callback,
            final SendersResponseListener listener)
        {
            super(callback);
            this.listener = listener;
        }

        @Override
        protected SendersResponse convertResult(final JsonObject result) {
            try {
                return convert(result.asMap());
            } catch (JsonException e) {
                e.addSuppressed(
                    new JsonException(
                        "Failed to parse response: "
                        + JsonType.HUMAN_READABLE.toString(result)));
                failed(e);
                return null;
            }
        }

        private SendersResponse convert(final JsonMap map)
            throws JsonException
        {
            JsonObject senderMlFeatures = map.get("sender_ml_features");
            if (senderMlFeatures != JsonNull.INSTANCE) {
                senderMlFeatures =
                    TypesafeValueContentHandler.parse(
                        senderMlFeatures.asString());
            }
            JsonObject senderMlEmbeddings = map.get("sender_ml_embeddings");
            if (senderMlEmbeddings != JsonNull.INSTANCE) {
                senderMlEmbeddings =
                    TypesafeValueContentHandler.parse(
                        senderMlEmbeddings.asString());
            }
            JsonList list = map.get("results").asList();
            int size = list.size();
            SendersResponse senders = new SendersResponse(
                size,
                senderMlFeatures,
                senderMlEmbeddings);
            for (int i = 0; i < size; ++i) {
                SenderResponse sender =
                    new SenderResponse(list.get(i).asMap());
                senders.add(sender);
                listener.personalFilterResult(
                    sender.uid(),
                    sender.pfilterLastType());
            }
            return senders;
        }
    }
}

