package ru.yandex.mail.so.factors.senders;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.LongPair;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.HashMapFactory;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.mail.so.api.v1.EmailInfo;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.LongSoFactorType;
import ru.yandex.mail.so.factors.types.MailMetaSoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.mail.senders.SenderInfo;
import ru.yandex.parser.mail.senders.SendersContext;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.stater.FixedSetStater;
import ru.yandex.util.string.StringUtils;

public class SendersExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Arrays.asList(
            MailMetaSoFactorType.MAIL_META,
            SmtpEnvelopeSoFactorType.SMTP_ENVELOPE,
            LongSoFactorType.LONG);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(SendersSoFactorType.SENDERS);

    private static final Pattern MSG_ID_PATTERN =
        Pattern.compile(
            "<("
            + "[^>]{25,}"
            + '|'
            + "[^>]*[0-9]+[A-Za-z]+[^>]*[0-9]+[A-Za-z]+[^>]*"
            + '|'
            + "[^>]*[a-zA-Z]+[0-9]+[^>]*[A-Za-z]+[0-9]+[^>]*"
            + '|'
            + "[^>]*[0-9]{10,}[^>]*"
            + ")>");

    private final SendersClient client;
    private final TimeFrameQueue<String> sendersResolutions;
    private final int maxReferences;

    public SendersExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableHttpHostConfig hostConfig =
            new HttpHostConfigBuilder(config).build();
        client =
            context.asyncClientRegistrar().registerClient(
                name,
                new SendersClient(
                context.asyncClientRegistrar().reactor(),
                hostConfig),
                hostConfig);
        maxReferences =
            config.get("max-references", PositiveIntegerValidator.INSTANCE);
        sendersResolutions = new TimeFrameQueue<>(context.metricsTimeFrame());
        context.registry().statersRegistrar().registerStater(
            new FixedSetStater<>(
                sendersResolutions,
                HashMapFactory.instance(),
                Arrays.asList("spam", "ham", "null"),
                name + "-pfilters-resolution-",
                "pfilters",
                name + " personal filters",
                null,
                1));
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {
        MailMetaInfo meta = inputs.get(0, MailMetaSoFactorType.MAIL_META);
        if (meta == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        SmtpEnvelopeHolder envelope =
            inputs.get(1, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
        if (envelope == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        Long senderUid = inputs.get(2, LongSoFactorType.LONG);
        String headers =
            StringUtils.concat('\n', meta.getLocal(MailMetaInfo.HEADERS), '\n');
        String senderHost = meta.receivedChainParser().sourceDomain(envelope);
        List<SenderInfo> sendersAddrs =
            new SendersContext(headers).extractSenders();
        if (context.logger().isLoggable(Level.INFO)) {
            context.logger().info(
                "Sender host: " + senderHost
                + ", sender addrs: " + sendersAddrs);
        }
        List<EmailInfo> recipients = envelope.recipients();
        int size = recipients.size();
        List<LongPair<Long>> users = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) {
            EmailInfo recipient = recipients.get(i);
            if (recipient.hasUid()) {
                long uid = recipient.getUid().getValue();
                Long suid;
                if (recipient.hasSuid()) {
                    suid = recipient.getSuid().getValue();
                } else {
                    if (context.logger().isLoggable(Level.INFO)) {
                        context.logger().info("No suid for uid " + uid);
                    }
                    suid = null;
                }
                users.add(new LongPair<>(uid, suid));
            } else if (context.logger().isLoggable(Level.INFO)) {
                context.logger().info(
                    "Recipient uid missing for: " + recipient);
            }
        }
        if (users.isEmpty()) {
            context.logger().info("Empty users, skipping senders request");
            callback.completed(NULL_RESULT);
            return;
        }
        List<String> inReplyTo =
            extractMsgIds(headers, "\nin-reply-to: ", maxReferences);
        List<String> references =
            extractMsgIds(headers, "\nreferences: ", maxReferences);
        SendersClient client = this.client.adjust(context.httpContext());
        try {
            client.sendersInfo(
                users,
                sendersAddrs,
                Objects.toString(senderHost),
                senderUid,
                inReplyTo,
                references,
                new SendersResponseStatsReporter(
                    sendersResolutions,
                    context.logger()),
                context.requestsListener().createContextGeneratorFor(client),
                new Callback(callback));
        } catch (BadRequestException e) {
            callback.failed(e);
        }
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        SendersExtractorFactory.INSTANCE.registerInternals(registry);
    }

    // Needle must be in the form '\n' + <lowercase-header-name> + ": "
    private static List<String> extractMsgIds(
        final String headers,
        final String needle,
        final int maxReferences)
    {
        int idx = headers.indexOf(needle);
        if (idx == -1) {
            return Collections.emptyList();
        } else {
            int start = idx + needle.length();
            int end = headers.indexOf('\n', start);
            String value = headers.substring(start, end);
            Matcher matcher = MSG_ID_PATTERN.matcher(value);
            if (matcher.find()) {
                List<String> result = new ArrayList<>(maxReferences);
                result.add(matcher.group());
                while (result.size() < maxReferences && matcher.find()) {
                    result.add(matcher.group());
                }
                return result;
            } else {
                return Collections.emptyList();
            }
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<
            SendersResponse,
            List<SoFactor<?>>>
    {
        Callback(final FutureCallback<? super List<SoFactor<?>>> callback) {
            super(callback);
        }

        @Override
        public void completed(final SendersResponse response) {
            callback.completed(
                Collections.singletonList(
                    SendersSoFactorType.SENDERS.createFactor(response)));
        }
    }
}

