package ru.yandex.mail.so.factors.senders;

import java.util.List;

import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.PositionSavingContainerFactory;
import ru.yandex.mail.so.factors.FactorsAccessViolationHandler;
import ru.yandex.mail.so.factors.SoFactorFieldAccessor;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;

public enum SendersSoFactorType implements SoFactorType<SendersResponse> {
    SENDERS;

    @Override
    public SendersResponse cast(final Object value) {
        if (value instanceof SendersResponse) {
            return (SendersResponse) value;
        } else {
            return null;
        }
    }

    @Override
    public SoFactorFieldAccessor fieldAccessorFor(final String fieldName) {
        switch (fieldName) {
            case "sender_info":
                return SenderInfoAccessor.INSTANCE;
            case "sender_ml_features":
                return MlFeaturesAccessor.INSTANCE;
            case "sender_ml_embeddings":
                return MlEmbeddingsAccessor.INSTANCE;
            default:
                return null;
        }
    }

    private enum SenderInfoAccessor implements SoFactorFieldAccessor {
        INSTANCE;

        @Override
        public SoFactorType<?> variableType() {
            return SendersSoFactorType.SENDERS;
        }

        @Override
        public SoFactorType<?> fieldType() {
            return JsonObjectSoFactorType.JSON_OBJECT;
        }

        private static void putString(
            final JsonMap map,
            final String name,
            final String value)
        {
            if (value != null) {
                map.put(name, new JsonString(value));
            }
        }

        private static void putStringList(
            final JsonMap map,
            final String name,
            final List<String> list)
        {
            int size = list.size();
            if (size > 0) {
                JsonList result = new JsonList(map.containerFactory(), size);
                for (int i = 0; i < size; ++i) {
                    result.add(new JsonString(list.get(i)));
                }
                map.put(name, result);
            }
        }

        @Override
        public JsonList extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof SendersResponse) {
                SendersResponse senders = (SendersResponse) value;
                int size = senders.size();
                JsonList result = new JsonList(
                    PositionSavingContainerFactory.INSTANCE,
                    size);

                for (int i = 0; i < size; ++i) {
                    SenderResponse sender = senders.get(i);
                    JsonMap map =
                        new JsonMap(PositionSavingContainerFactory.INSTANCE);
                    result.add(map);

                    JsonMap request =
                        new JsonMap(PositionSavingContainerFactory.INSTANCE);
                    map.put("request", request);
                    request.put(
                        "uid",
                        new JsonString(Long.toString(sender.uid())));
                    Long suid = sender.suid();
                    if (suid != null) {
                        request.put(
                            "suid",
                            new JsonString(Long.toString(suid)));
                    }
                    putString(request, "email", sender.email());
                    putString(request, "domain", sender.domain());
                    putString(request, "sender_host", sender.senderHost());
                    putStringList(request, "in-reply-to", sender.inReplyTo());
                    putStringList(request, "references", sender.references());

                    JsonMap response =
                        new JsonMap(PositionSavingContainerFactory.INSTANCE);
                    map.put("response", response);
                    response.put(
                        "sent_count",
                        new JsonString(Long.toString(sender.sentCount())));
                    response.put(
                        "received_count",
                        new JsonString(Long.toString(sender.receivedCount())));
                    response.put(
                        "received_read_count",
                        new JsonString(
                            Long.toString(sender.receivedReadCount())));
                    response.put(
                        "last_contacted",
                        new JsonString(Long.toString(sender.lastContacted())));

                    response.put(
                        "domain_sent_count",
                        new JsonString(
                            Long.toString(sender.domainSentCount())));
                    response.put(
                        "domain_received_count",
                        new JsonString(
                            Long.toString(sender.domainReceivedCount())));
                    response.put(
                        "domain_received_read_count",
                        new JsonString(
                            Long.toString(sender.domainReceivedReadCount())));

                    putString(
                        response,
                        "pfilter_last_type",
                        sender.pfilterLastType());
                    response.put(
                        "pfilter_spams",
                        new JsonString(Long.toString(sender.pfilterSpams())));
                    response.put(
                        "pfilter_hams",
                        new JsonString(Long.toString(sender.pfilterHams())));

                    putString(
                        response,
                        "tabpf_last_tab",
                        sender.tabPfLastTab());

                    JsonObject userMlFeatures = sender.userMlFeatures();
                    if (userMlFeatures != JsonNull.INSTANCE) {
                        response.put("user_ml_features", userMlFeatures);
                    }

                    JsonObject userMlEmbeddings = sender.userMlEmbeddings();
                    if (userMlEmbeddings != JsonNull.INSTANCE) {
                        response.put("user_ml_embeddings", userMlEmbeddings);
                    }

                    response.put(
                        "in_reply_to_matches",
                        JsonBoolean.valueOf(sender.inReplyToMatches()));
                    response.put(
                        "references_matches",
                        JsonBoolean.valueOf(sender.referencesMatches()));
                }
                return result;
            }
            return null;
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(".sender_info");
        }
    }

    private static JsonObject unmaskNull(final JsonObject value) {
        if (value == JsonNull.INSTANCE) {
            return null;
        } else {
            return value;
        }
    }

    private enum MlFeaturesAccessor implements SoFactorFieldAccessor {
        INSTANCE;

        @Override
        public SoFactorType<?> variableType() {
            return SendersSoFactorType.SENDERS;
        }

        @Override
        public SoFactorType<?> fieldType() {
            return JsonObjectSoFactorType.JSON_OBJECT;
        }

        @Override
        public JsonObject extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof SendersResponse) {
                return unmaskNull(((SendersResponse) value).senderMlFeatures());
            }
            return null;
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(".sender_ml_features");
        }
    }

    private enum MlEmbeddingsAccessor implements SoFactorFieldAccessor {
        INSTANCE;

        @Override
        public SoFactorType<?> variableType() {
            return SendersSoFactorType.SENDERS;
        }

        @Override
        public SoFactorType<?> fieldType() {
            return JsonObjectSoFactorType.JSON_OBJECT;
        }

        @Override
        public JsonObject extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof SendersResponse) {
                return unmaskNull(
                    ((SendersResponse) value).senderMlEmbeddings());
            }
            return null;
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            sb.append(".sender_ml_embeddings");
        }
    }
}

