package ru.yandex.mail.so.factors.sherlock;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import javax.annotation.concurrent.ThreadSafe;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.so.api.v1.EmailInfo;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorFactoryContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.JsonObjectSoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.mail.so.factors.types.TikaiteDocSoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.util.string.StringUtils;

@ThreadSafe
public class TemplateMasterExtractor implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
        Arrays.asList(
            TikaiteDocSoFactorType.TIKAITE_DOC,
            SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
    private static final List<SoFactorType<?>> OUTPUTS =
        Collections.singletonList(JsonObjectSoFactorType.JSON_OBJECT);
    private static final CollectionParser<String, List<String>, Exception>
        FIELDS_LIST_PARSER = new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            ArrayList::new);

    private final TemplateMasterClient templateMasterClient;
    private final List<String> tikaiteFieldPriority;

    public TemplateMasterExtractor(
        final String name,
        final SoFactorsExtractorFactoryContext context,
        final IniConfig config)
        throws ConfigException
    {
        ImmutableHttpHostConfig hostConfig =
            new HttpHostConfigBuilder(config).build();
        templateMasterClient =
            context.asyncClientRegistrar().registerClient(
                name,
                new TemplateMasterClient(
                    context.asyncClientRegistrar().reactor(),
                    hostConfig),
                hostConfig);
        tikaiteFieldPriority = config.get("body-fields", FIELDS_LIST_PARSER);
    }

    @Override
    public void close() {
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void extract(
        final SoFactorsExtractorContext context,
        final SoFunctionInputs inputs,
        final FutureCallback<? super List<SoFactor<?>>> callback)
    {

        JsonMap tikaiteDoc = inputs.get(0, TikaiteDocSoFactorType.TIKAITE_DOC);
        if (tikaiteDoc == null) {
            callback.completed(NULL_RESULT);
            return;
        }
        String body = null;
        for (String field: tikaiteFieldPriority) {
            body = tikaiteDoc.getLastOrNull(field);
            if (body != null) {
                break;
            }
        }
        if (body == null) {
            context.logger().info("No body found");
            callback.completed(NULL_RESULT);
            return;
        }

        SmtpEnvelopeHolder envelope =
            inputs.get(1, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);

        Map.Entry<String, String> from = extractFrom(tikaiteDoc);
        if (from == null
            || (from.getValue() == null && from.getKey() == null))
        {
            if (envelope != null
                && envelope.from() != null
                && envelope.from().getEmail() != null)
            {
                from = MailAliases.INSTANCE.parseAndNormalize(
                    envelope.from().getEmail());
            }
            if (from == null
                || (from.getValue() == null && from.getKey() == null))
            {
                context.logger().info("From not found, aborting");
                callback.completed(NULL_RESULT);
                return;
            } else {
                // Actually, works 28% of times. Investigation required
                context.logger().info("Fallback successful");
            }
        }
        String fromDomain = from.getValue();
        if (fromDomain != null) {
            int idx = fromDomain.lastIndexOf('.');
            if (idx != -1) {
                idx = fromDomain.lastIndexOf('.', idx - 1);
                if (idx != -1) {
                    // truncate market.yandex.ru to yandex.ru
                    fromDomain =
                        MailAliases.INSTANCE.normalizeDomain(
                            fromDomain.substring(idx + 1));
                }
            }
        }
        // Email from yandex.ru, yandex.com, market.yandex.ru etc.
        // Not a scope for template master
        if ("yandex.ru".equals(fromDomain)
            || "yandex-team.ru".equals(fromDomain))
        {
            context.logger().info("Email from yandex, skipping");
            callback.completed(NULL_RESULT);
            return;
        }

        StringWriter attributes = new StringWriter();
        try (JsonWriter attrWriter =
            JsonType.NORMAL.create(attributes)) {
            attrWriter.startObject();

            String fromString =
                StringUtils.concat(from.getKey(), '@', from.getValue());
            attrWriter.key("from");
            attrWriter.value(fromString);

            String subject = tikaiteDoc.getLastOrNull("hdr_subject");
            if (subject != null) {
                attrWriter.key("subject");
                attrWriter.value(subject);
            }

            if (envelope != null) {
                attrWriter.key("queueId");
                attrWriter.value(
                    envelope.envelope().getConnectInfo().getSessionId());

                attrWriter.key("uids");
                attrWriter.startArray();
                for (EmailInfo recipient : envelope.recipients()) {
                    if (recipient.hasUid()) {
                        attrWriter.value(recipient.getUid().getValue());
                    } else if (context.logger().isLoggable(Level.INFO)) {
                        context.logger().info(
                            "Recipient uid missing for: "
                                + recipient.getAddress().getEmail());
                    }
                }
                attrWriter.endArray();
            }
            attrWriter.endObject();
        } catch (IOException unreachable) {
            callback.failed(unreachable);
            return;
        }

        TemplateMasterClient adjustedClient =
            this.templateMasterClient.adjust(context.httpContext());
        try {
            adjustedClient.detemple(
                body,
                attributes.toString(),
                from.getValue() == null ? from.getKey() : from.getValue(),
                context.requestsListener()
                    .createContextGeneratorFor(adjustedClient),
                new Callback(callback));
        } catch (BadRequestException e) {
            callback.failed(e);
        }
    }

    private static Map.Entry<String, String> extractFrom(JsonMap tikaiteDoc) {
        String from;
        try {
            from = tikaiteDoc.get("hdr_from_normalized").asStringOrNull();
        } catch (JsonBadCastException e) {
            return null;
        }
        if (from == null) {
            return null;
        }
        from = StringUtils.firstLine(from).trim();
        return MailAliases.INSTANCE.parseAndNormalize(from);
    }

    @Override
    public void registerInternals(final SoFactorsExtractorsRegistry registry)
        throws ConfigException
    {
        // Input: body, subject, from
        registry.typesRegistry().registerSoFactorType(
            TikaiteDocSoFactorType.TIKAITE_DOC);
        // Input: qid and uids
        registry.typesRegistry().registerSoFactorType(
            SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
        // Output: { delta: <LIST OF TOKENS>,
        //           stable_sign: <long>, donor_and_contains_urls: <bool>}
        registry.typesRegistry().registerSoFactorType(
            JsonObjectSoFactorType.JSON_OBJECT);
    }

    private static class Callback
        extends AbstractFilterFutureCallback<
            TemplateMasterResponse,
            List<SoFactor<?>>>
    {
        Callback(final FutureCallback<? super List<SoFactor<?>>> callback) {
            super(callback);
        }

        @Override
        public void completed(final TemplateMasterResponse response) {
            if (response.hasTemplate()) {
                try {
                    JsonMap map =
                        new JsonMap(BasicContainerFactory.INSTANCE, 6);
                    List<String> delta = response.delta();
                    int size = delta.size();
                    JsonList deltaRes =
                        new JsonList(BasicContainerFactory.INSTANCE, size);
                    for (String s : delta) {
                        deltaRes.add(new JsonString(s));
                    }
                    map.put("delta", deltaRes);
                    Long stableSign = response.stableSign();
                    if (stableSign != null) {
                        map.put("stable_sign", new JsonLong(stableSign));
                    }
                    callback.completed(
                        Collections.singletonList(
                            JsonObjectSoFactorType.JSON_OBJECT.createFactor(
                                map)));
                } catch (Exception e) {
                    failed(e);
                }
            } else {
                callback.completed(NULL_RESULT);
            }
        }
    }
}
