package ru.yandex.mail.so.factors.types;

import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.so.factors.FactorsAccessViolationHandler;
import ru.yandex.mail.so.factors.SoFactorFieldAccessorBase;

public enum JsonObjectSoFactorType implements SoFactorType<JsonObject> {
    JSON_OBJECT;

    @Override
    public JsonObject cast(final Object value) {
        if (value instanceof JsonObject) {
            return (JsonObject) value;
        } else {
            return null;
        }
    }

    @Override
    public SoFactorFieldAccessorBase fieldAccessorFor(final String fieldName) {
        return fieldAccessorFor(this, fieldName);
    }

    public static SoFactorFieldAccessorBase fieldAccessorFor(
        final SoFactorType<?> variableType,
        final String fieldName)
    {
        switch (fieldName) {
            case "__boolean__":
                return new BooleanAccessor(variableType);
            case "__long__":
                return new LongAccessor(variableType);
            case "__double__":
                return new DoubleAccessor(variableType);
            case "__string__":
                return new StringAccessor(variableType);
            // JsonObject conversions
            case "__json_list__":
                return new JsonListAccessor(variableType);
            case "__json_map__":
                return new JsonMapAccessor(variableType);
            case "__json_object__":
                return new JsonObjectAccessor(variableType);
            default:
                return new JsonFieldAccessor(variableType, fieldName);
        }
    }

    @Override
    public SoFactorFieldAccessorBase fieldAccessorFor(final int fieldIndex) {
        return new IndexAccessor(this, fieldIndex);
    }

    public static class BooleanAccessor extends SoFactorFieldAccessorBase {
        public BooleanAccessor(final SoFactorType<?> variableType) {
            super(variableType, BooleanSoFactorType.BOOLEAN, ".__boolean__");
        }

        @Override
        public Boolean extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof JsonObject) {
                try {
                    return ((JsonObject) value).asBoolean();
                } catch (JsonBadCastException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + JsonType.NORMAL.toString(value)
                        + " to boolean",
                        e);
                }
            }
            return null;
        }
    }

    public static class LongAccessor extends SoFactorFieldAccessorBase {
        public LongAccessor(final SoFactorType<?> variableType) {
            super(variableType, LongSoFactorType.LONG, ".__long__");
        }

        @Override
        public Long extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler) {
            if (value instanceof JsonObject) {
                try {
                    return ((JsonObject) value).asLong();
                } catch (JsonBadCastException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + JsonType.NORMAL.toString(value)
                        + " to long",
                        e);
                }
            }
            return null;
        }
    }

    public static class DoubleAccessor extends SoFactorFieldAccessorBase {
        public DoubleAccessor(final SoFactorType<?> variableType) {
            super(variableType, DoubleSoFactorType.DOUBLE, ".__double__");
        }

        @Override
        public Double extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof JsonObject) {
                try {
                    return ((JsonObject) value).asDouble();
                } catch (JsonBadCastException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + JsonType.NORMAL.toString(value)
                        + " to double",
                        e);
                }
            }
            return null;
        }
    }

    public static class StringAccessor extends SoFactorFieldAccessorBase {
        public StringAccessor(final SoFactorType<?> variableType) {
            super(variableType, StringSoFactorType.STRING, ".__string__");
        }

        @Override
        public String extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof JsonObject) {
                try {
                    return ((JsonObject) value).asString();
                } catch (JsonBadCastException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + JsonType.NORMAL.toString(value)
                        + " to String",
                        e);
                }
            }
            return null;
        }
    }

    public static class JsonListAccessor extends SoFactorFieldAccessorBase {
        public JsonListAccessor(final SoFactorType<?> variableType) {
            super(
                variableType,
                JsonListSoFactorType.JSON_LIST,
                ".__json_list__");
        }

        @Override
        public JsonList extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof JsonList) {
                return (JsonList) value;
            } else if (value != null) {
                accessViolationHandler.handleFieldConversionFailure(
                    "Can't cast " + JsonType.NORMAL.toString(value)
                    + " to JsonList");
            }
            return null;
        }
    }

    public static class JsonMapAccessor extends SoFactorFieldAccessorBase {
        public JsonMapAccessor(final SoFactorType<?> variableType) {
            super(variableType, JsonMapSoFactorType.JSON_MAP, ".__json_map__");
        }

        @Override
        public JsonMap extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof JsonMap) {
                return (JsonMap) value;
            } else if (value != null) {
                accessViolationHandler.handleFieldConversionFailure(
                    "Can't cast " + JsonType.NORMAL.toString(value)
                    + " to JsonMap");
            }
            return null;
        }
    }

    public static class JsonObjectAccessor extends SoFactorFieldAccessorBase {
        public JsonObjectAccessor(final SoFactorType<?> variableType) {
            super(
                variableType,
                JsonObjectSoFactorType.JSON_OBJECT,
                ".__json_object__");
        }

        @Override
        public JsonObject extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof JsonObject) {
                return (JsonObject) value;
            }
            return null;
        }
    }

    public static class JsonFieldAccessor extends SoFactorFieldAccessorBase {
        private final String fieldName;

        public JsonFieldAccessor(
            final SoFactorType<?> variableType,
            final String fieldName)
        {
            super(
                variableType,
                JsonObjectSoFactorType.JSON_OBJECT,
                '.' + fieldName);
            this.fieldName = fieldName;
        }

        @Override
        public JsonObject extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            JsonObject result = null;
            if (value instanceof JsonMap) {
                result = ((JsonMap) value).get(fieldName);
                if (result == JsonNull.INSTANCE) {
                    return null;
                }
            }
            return result;
        }
    }

    public static class IndexAccessor extends SoFactorFieldAccessorBase {
        private final int fieldIndex;

        public IndexAccessor(
            final SoFactorType<?> variableType,
            final int fieldIndex)
        {
            super(
                variableType,
                JsonObjectSoFactorType.JSON_OBJECT,
                "[" + fieldIndex + ']');
            this.fieldIndex = fieldIndex;
        }

        @Override
        public JsonObject extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            JsonObject result = null;
            if (value instanceof JsonList) {
                JsonList list = (JsonList) value;
                if (list.size() > fieldIndex) {
                    result = list.get(fieldIndex);
                    if (result == JsonNull.INSTANCE) {
                        result = null;
                    }
                }
            }
            return result;
        }
    }
}

