package ru.yandex.mail.so.factors.types;

import ru.yandex.json.dom.JsonString;
import ru.yandex.mail.so.factors.FactorsAccessViolationHandler;
import ru.yandex.mail.so.factors.SoFactorFieldAccessorBase;
import ru.yandex.parser.string.BooleanParser;

public enum StringSoFactorType implements SoFactorType<String> {
    STRING;

    @Override
    public String cast(final Object value) {
        if (value instanceof String) {
            return (String) value;
        } else {
            return null;
        }
    }

    @Override
    public SoFactorFieldAccessorBase fieldAccessorFor(final String fieldName) {
        return fieldAccessorFor(this, fieldName);
    }

    public static SoFactorFieldAccessorBase fieldAccessorFor(
        final SoFactorType<?> variableType,
        final String fieldName)
    {
        switch (fieldName) {
            case "__boolean__":
                return new BooleanAccessor(variableType);
            case "__double__":
                return new DoubleAccessor(variableType);
            case "__long__":
                return new LongAccessor(variableType);
            case "__size__":
                return new SizeAccessor(variableType);
            case "__json_object__":
                return new JsonObjectAccessor(variableType);
            default:
                return null;
        }
    }

    public static class BooleanAccessor extends SoFactorFieldAccessorBase {
        public BooleanAccessor(final SoFactorType<?> variableType) {
            super(variableType, BooleanSoFactorType.BOOLEAN, ".__boolean__");
        }

        @Override
        public Boolean extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof String) {
                try {
                    return BooleanParser.INSTANCE.apply((String) value);
                } catch (RuntimeException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + value + " to boolean",
                        e);
                }
            }
            return null;
        }
    }

    public static class DoubleAccessor extends SoFactorFieldAccessorBase {
        public DoubleAccessor(final SoFactorType<?> variableType) {
            super(variableType, DoubleSoFactorType.DOUBLE, ".__double__");
        }

        @Override
        public Double extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof String) {
                try {
                    return Double.valueOf((String) value);
                } catch (RuntimeException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + value + " to double",
                        e);
                }
            }
            return null;
        }
    }

    public static class LongAccessor extends SoFactorFieldAccessorBase {
        public LongAccessor(final SoFactorType<?> variableType) {
            super(variableType, LongSoFactorType.LONG, ".__long__");
        }

        @Override
        public Long extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof String) {
                try {
                    return Long.valueOf((String) value);
                } catch (RuntimeException e) {
                    accessViolationHandler.handleFieldConversionFailure(
                        "Can't cast " + value + " to long",
                        e);
                }
            }
            return null;
        }
    }

    public static class SizeAccessor extends SoFactorFieldAccessorBase {
        public SizeAccessor(final SoFactorType<?> variableType) {
            super(variableType, LongSoFactorType.LONG, ".__size__");
        }

        @Override
        public Long extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof String) {
                return (long) ((String) value).length();
            } else {
                return null;
            }
        }
    }

    public static class JsonObjectAccessor extends SoFactorFieldAccessorBase {
        public JsonObjectAccessor(final SoFactorType<?> variableType) {
            super(
                variableType,
                JsonObjectSoFactorType.JSON_OBJECT,
                ".__json_object__");
        }

        @Override
        public JsonString extractField(
            final Object value,
            final FactorsAccessViolationHandler accessViolationHandler)
        {
            if (value instanceof String) {
                return new JsonString((String) value);
            }
            return null;
        }
    }
}

