#include "jniwrapper_base.h"

#include <util/charset/wide.h>
#include <util/generic/bt_exception.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>

#include <stdexcept>

int NJniWrapper::ProcessJniWrapperException(char** out) noexcept {
    try {
        try {
            throw;
        } catch (const std::bad_alloc& e) {
            throw;
        } catch (const std::invalid_argument& e) {
            *out = strdup(CurrentExceptionMessage().c_str());
            return -2;
        } catch (...) {
            *out = strdup(CurrentExceptionMessage().c_str());
            return -1;
        }
    } catch (const std::bad_alloc& e) {
    }
    *out = nullptr;
    return -1;
}

int NJniWrapper::ProcessJniWrapperException(wchar16** out, size_t* outLen) noexcept {
    try {
        try {
            throw;
        } catch (const std::bad_alloc& e) {
            throw;
        } catch (const std::invalid_argument& e) {
            *out = Utf16Dup(UTF8ToWide<true>(CurrentExceptionMessage()), *outLen);
            return -2;
        } catch (...) {
            *out = Utf16Dup(UTF8ToWide<true>(CurrentExceptionMessage()), *outLen);
            return -1;
        }
    } catch (const std::bad_alloc& e) {
    }
    *out = nullptr;
    *outLen = 0;
    return -1;
}

wchar16* NJniWrapper::Utf16Dup(const TUtf16String& str, size_t& len) noexcept {
    len = str.length();
    wchar16* result = (wchar16*) malloc(len * sizeof(wchar16));
    if (result) {
        memcpy(result, str.data(), len * sizeof(wchar16));
    }
    return result;
}

void NJniWrapper::Throw(
    JNIEnv* env,
    const char* exceptionClassName,
    const char* message)
    noexcept
{
    if (!env->ExceptionOccurred()) {
        jclass exceptionClass = env->FindClass(exceptionClassName);
        if (exceptionClass) {
            env->ThrowNew(exceptionClass, message);
        }
    }
}

void NJniWrapper::Throw(
    JNIEnv* env,
    const char* exceptionClassName,
    const TString& message)
    noexcept
{
    Throw(env, exceptionClassName, message.c_str());
}

void NJniWrapper::Throw(
    JNIEnv* env,
    const char* exceptionClassName,
    const wchar16* message,
    const size_t messageLen)
    noexcept
{
    try {
        TString utf8Message = WideToUTF8(message, messageLen);
        Throw(env, exceptionClassName, utf8Message.c_str());
    } catch (...) {
        RethrowAsJavaException(env);
    }
}

void NJniWrapper::Throw(
    JNIEnv* env,
    const char* exceptionClassName,
    const TUtf16String& message)
    noexcept
{
    Throw(env, exceptionClassName, message.Data(), message.Size());
}

void NJniWrapper::RethrowAsJavaException(JNIEnv* env) noexcept {
    try {
        try {
            Throw(
                env,
                "java/lang/RuntimeException",
                CurrentExceptionMessage().c_str());
        } catch (const std::exception& e) {
            Throw(
                env,
                "java/lang/RuntimeException",
                e.what());
        }
    } catch (...) {
        Throw(
            env,
            "java/lang/RuntimeException",
            "Something weird happened");
    }
}

struct TJStringHolder {
    JNIEnv* const Env;
    jstring Str;
    const size_t Size;
    const wchar16* Chars;

    TJStringHolder(JNIEnv* env, jstring str)
        : Env(env)
        , Str(str)
        , Size(static_cast<size_t>(Env->GetStringLength(Str)))
        , Chars(reinterpret_cast<const wchar16*>(Env->GetStringCritical(Str, nullptr)))
    {
        if (!Chars) {
            ythrow TWithBackTrace<yexception>()
                << "Failed to retrieve string chars";
        }
    }

    ~TJStringHolder() noexcept {
        Env->ReleaseStringCritical(Str, reinterpret_cast<const jchar*>(Chars));
    }
};

TUtf16String NJniWrapper::ConvertJString(JNIEnv* env, jstring str) {
    TJStringHolder string{env, str};
    return TUtf16String{string.Chars, string.Size};
}

TString NJniWrapper::JStringToUtf(JNIEnv* env, jstring str) {
    TJStringHolder string{env, str};
    return WideToUTF8(string.Chars, string.Size);
}

jstring NJniWrapper::UtfToJString(JNIEnv* env, const char* str) {
    TUtf16String utf16Str{UTF8ToWide<true>(str)};
    return env->NewString((const jchar*) utf16Str.Data(), utf16Str.Size());
}

NJniWrapper::TFloatArrayElements::TFloatArrayElements(
    JNIEnv* env,
    jfloatArray data,
    jsize size,
    jint mode)
    : Env(env)
    , JavaData(data)
    , Size(size)
    , Mode(mode)
{
    jboolean copy;
    Data = Env->GetFloatArrayElements(JavaData, &copy);
    if (!Data) {
        ythrow TWithBackTrace<yexception>()
            << "Failed to GetFloatArrayElements";
    }
    Copy = copy == JNI_TRUE;
}

NJniWrapper::TByteArrayElements::TByteArrayElements(
    JNIEnv* env,
    jbyteArray data,
    jsize size,
    jint mode)
    : Env(env)
    , JavaData(data)
    , Size(size)
    , Mode(mode)
{
    jboolean copy;
    Data = Env->GetByteArrayElements(JavaData, &copy);
    if (!Data) {
        ythrow TWithBackTrace<yexception>()
            << "Failed to GetByteArrayElements";
    }
    Copy = copy == JNI_TRUE;
}

void JniWrapperFree(void* ptr) noexcept {
    free(ptr);
}

