#pragma once

#include <util/generic/bt_exception.h>
#include <util/generic/fwd.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/system/types.h>

#include <jni.h>

namespace NJniWrapper {
    int ProcessJniWrapperException(char** out) noexcept;
    int ProcessJniWrapperException(wchar16** out, size_t* outLen) noexcept;
    wchar16* Utf16Dup(const TUtf16String& str, size_t& len) noexcept;
    void Throw(
        JNIEnv* env,
        const char* exceptionClassName,
        const char* message)
        noexcept;
    void Throw(
        JNIEnv* env,
        const char* exceptionClassName,
        const TString& message)
        noexcept;
    void Throw(
        JNIEnv* env,
        const char* exceptionClassName,
        const wchar16* message,
        const size_t messageLen)
        noexcept;
    void Throw(
        JNIEnv* env,
        const char* exceptionClassName,
        const TUtf16String& message)
        noexcept;
    // Converts C++ exception to Java RuntimeException
    void RethrowAsJavaException(JNIEnv* env) noexcept;
    // Throws exception on error
    TUtf16String ConvertJString(JNIEnv* env, jstring str);
    // Throws exception on error
    TString JStringToUtf(JNIEnv* env, jstring str);
    // Throws exception on error, returns NULL if env->NewString failed
    jstring UtfToJString(JNIEnv* env, const char* str);

    class TFloatArrayElements {
    private:
        JNIEnv* const Env;
        jfloatArray JavaData;
        const size_t Size;
        float* Data;
        bool Copy;
        jint Mode;

    public:
        TFloatArrayElements(JNIEnv* env, jfloatArray data, jint mode)
            : TFloatArrayElements(env, data, env->GetArrayLength(data), mode)
        {
        }

        TFloatArrayElements(JNIEnv* env, jfloatArray data, jsize size, jint mode);

        ~TFloatArrayElements() noexcept {
            Env->ReleaseFloatArrayElements(JavaData, Data, Mode);
        }

        size_t GetSize() const {
            return Size;
        }

        float* GetData() {
            return Data;
        }

        const float* GetData() const {
            return Data;
        }

        bool IsCopy() const {
            return Copy;
        }

        jint GetMode() const {
            return Mode;
        }

        void SetMode(jint mode) {
            Mode = mode;
        }
    };

    class TByteArrayElements {
    private:
        JNIEnv* const Env;
        jbyteArray JavaData;
        const size_t Size;
        signed char* Data;
        bool Copy;
        jint Mode;

    public:
        TByteArrayElements(JNIEnv* env, jbyteArray data, jint mode)
            : TByteArrayElements(env, data, env->GetArrayLength(data), mode)
        {
        }

        TByteArrayElements(JNIEnv* env, jbyteArray data, jsize size, jint mode);

        ~TByteArrayElements() noexcept {
            Env->ReleaseByteArrayElements(JavaData, Data, Mode);
        }

        size_t GetSize() const {
            return Size;
        }

        signed char* GetData() {
            return Data;
        }

        const signed char* GetData() const {
            return Data;
        }

        bool IsCopy() const {
            return Copy;
        }

        jint GetMode() const {
            return Mode;
        }

        void SetMode(jint mode) {
            Mode = mode;
        }
    };

    template <typename T>
    class TCriticalArrayElements {
    private:
        JNIEnv* const Env;
        jarray JavaData;
        const size_t Size;
        T* Data;
        bool Copy;
        jint Mode;

    public:
        TCriticalArrayElements(JNIEnv* env, jarray data, jint mode)
            : TCriticalArrayElements(env, data, env->GetArrayLength(data), mode)
        {
        }

        TCriticalArrayElements(JNIEnv* env, jarray data, jsize size, jint mode)
            : Env(env)
            , JavaData(data)
            , Size(size)
            , Mode(mode)
        {
            jboolean copy;
            Data = static_cast<T*>(Env->GetPrimitiveArrayCritical(JavaData, &copy));
            if (!Data) {
                ythrow TWithBackTrace<yexception>()
                    << "Failed to GetFloatArrayElements";
            }
            Copy = copy == JNI_TRUE;
        }

        ~TCriticalArrayElements() noexcept {
            Env->ReleasePrimitiveArrayCritical(JavaData, Data, Mode);
        }

        size_t GetSize() const {
            return Size;
        }

        T* GetData() {
            return Data;
        }

        const T* GetData() const {
            return Data;
        }

        bool IsCopy() const {
            return Copy;
        }

        jint GetMode() const {
            return Mode;
        }

        void SetMode(jint mode) {
            Mode = mode;
        }
    };
}

// Calls free() on pointer
// Used to release strings strdup'ed by process function
extern "C" void JniWrapperFree(void* ptr) noexcept;

