#pragma once

#include <util/generic/hash.h>
#include <util/generic/utility.h>

template <class Key, class Value, class HashFcn = THash<Key>, class EqualKey = TEqualTo<Key>, class Alloc = std::allocator<Value>>
class TDoubleBarrelLRUCache {
private:
    typedef THashMap<Key, Value, HashFcn, EqualKey, Alloc> TMapType;
    const size_t MaxSize;
    TMapType Barrel1;
    TMapType Barrel2;
    TMapType* Master;
    TMapType* Slave;

private:
    void CheckSizeLimit() {
        if (Master->size() >= MaxSize) {
            DoSwap(Master, Slave);
            Master->clear();
        }
    }

public:
    TDoubleBarrelLRUCache(size_t maxSize)
        : MaxSize(maxSize)
        , Master(&Barrel1)
        , Slave(&Barrel2)
    {
    }

    template <class TheKey>
    bool Get(const TheKey& key, Value& value) {
        auto iter = Master->find(key);
        if (iter == Master->end()) {
            iter = Slave->find(key);
            if (iter == Slave->end()) {
                return false;
            } else {
                Master->insert(*iter);
                // It is important to store type before size limit check,
                // because it could clear slave and invalidate iterator
                value = iter->second;
                CheckSizeLimit();
            }
        } else {
            value = iter->second;
        }
        return true;
    }

    template <class TheKey, class TheValue>
    void Put(const TheKey& key, const TheValue& value) {
        Master->emplace(key, value);
        CheckSizeLimit();
    }

    size_t Size() const noexcept {
        return Barrel1.size() + Barrel2.size();
    }
};

