#include "field_projectors.h"

#include <util/generic/iterator_range.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <library/cpp/iterator/mapped.h>
#include <iterator>
#include "factory.h"

namespace NLSA {
    TProjectorField::TProjectorField(TField field, TVector<TSimpleSharedPtr<const IProjector>> projectors) noexcept
            : Field(field), Projectors(std::move(projectors)) {
        TStringOutput s(Represantation);
        s << Field;
        for (const TSimpleSharedPtr<const IProjector> &p : Projectors) {
            s << '.' << p->GetRepresentation();
        }
    }

    [[nodiscard]] TVector<TString> TProjectorField::Apply(const TVector<TString> &words) const {
        if (!Projectors) {
            return words;
        } else {
            auto it = Projectors.cbegin();
            TVector<TString> tranformedWords = it->Get()->Apply(words);

            for (++it; it != Projectors.cend(); ++it) {
                tranformedWords = it->Get()->Apply(tranformedWords);
            }

            return tranformedWords;
        }
    }

    TField TProjectorField::GetField() const {
        return Field;
    }

    [[nodiscard]] const TString &TProjectorField::GetRepresentation() const {
        return Represantation;
    }

    TProjectorField TProjectorField::Parse(const TStringBuf &src) {
        const auto splitter = StringSplitter(src).Split('.').SkipEmpty();
        auto it = splitter.begin();

        const TField field = FromString(it->Token());
        ++it;

        TVector<TSimpleSharedPtr<const IProjector>> projectors;

        for (; it != splitter.end(); ++it) {
            projectors.emplace_back(ProjectorFactory(*it));
        }

        return TProjectorField(field, std::move(projectors));
    }
} // namespace NLSA
