#include <util/string/cast.h>
#include "segment.h"

#include <library/cpp/json/json_writer.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>
#include <util/string/split.h>
#include <util/string/join.h>
#include <util/generic/strbuf.h>

namespace NLSA {
    bool TSegmentProjector::Accept(const TStringBuf& funcName) noexcept {
        return funcName == NAME;
    }

    TSimpleSharedPtr<const TSegmentProjector> TSegmentProjector::Make(const TStringBuf& args) {
        size_t size, overlap, maxLen;
        StringSplitter(args).SplitBySet(", )").SkipEmpty().CollectInto(&size, &overlap, &maxLen);

        return MakeSimpleShared<const TSegmentProjector>(size, overlap, maxLen);
    }

    TSegmentProjector::TSegmentProjector(size_t size, size_t overlap, size_t maxLen) noexcept
        : Size(size)
        , Overlap(overlap)
        , MaxLen(maxLen)
        , Represantation(TString(NAME) + '(' + ToString(Size) + ',' + ToString(Overlap) + ',' + ToString(MaxLen) + ')')
    {
        Y_ENSURE(Overlap < Size);
        Y_ENSURE(MaxLen >= Size);
    }

    size_t TSegmentProjector::GetSize() const {
        return Size;
    }

    size_t TSegmentProjector::GetOverlap() const {
        return Overlap;
    }

    size_t TSegmentProjector::GetMaxLen() const {
        return MaxLen;
    }

    TVector<TString> TSegmentProjector::Apply(const TVector<TString>& words) const {
        TStringStream out, currentWord;

        NJson::TJsonWriter json(&out, false);
        json.OpenMap();
        json.WriteKey("segments");
        json.OpenArray();
        auto limit = Min(MaxLen, words.size());
        limit -= (limit < Size) ? 0 : Size;
        for (size_t i = 0; i <= limit; i += Size - Overlap) {
            for (size_t j = i; j < Min(i + Size, words.size(), MaxLen); j++) {
                if (j > i) {
                    currentWord << " ";
                }
                currentWord << words[j];
            }
            json.Write(currentWord.Str());
            currentWord.Clear();
        }

        json.CloseArray();
        json.CloseMap();
        json.Flush();

        return {std::move(out.Str())};
    }

    const TString& TSegmentProjector::GetRepresentation() const {
        return Represantation;
    }
}