#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/buffer.h>
#include <util/stream/buffer.h>
#include <mail/so/libs/lsa_projectors/field_projectors.h>
#include <mail/so/libs/lsa_projectors/factory.h>
#include <mail/so/libs/lsa_projectors/crop.h>
#include <mail/so/libs/lsa_projectors/segment.h>
#include <mail/so/libs/lsa_projectors/plainify.h>

using namespace NLSA;

Y_UNIT_TEST_SUITE(FieldProjectors) {
    Y_UNIT_TEST(ParseCrop) {
        TVector<TString> words;
        for (int i = 0; i < 60; i++) {
            words.push_back(ToString(i));
        }
        {
            const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("crop(30)");
            UNIT_ASSERT_VALUES_EQUAL(dynamic_cast<const TCropProjector*>(p.Get())->GetCount(), 30);

            TVector<TString> cropped;
            for (int i = 0; i < 30; i++) {
                cropped.push_back(ToString(i));
            }
            UNIT_ASSERT_VALUES_EQUAL(cropped, p->Apply(words));
        }
        {
            const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("crop(70)");
            UNIT_ASSERT_VALUES_EQUAL(dynamic_cast<const TCropProjector*>(p.Get())->GetCount(), 70);

            UNIT_ASSERT_VALUES_EQUAL(words, p->Apply(words));
        }
    }
    Y_UNIT_TEST(ParseField) {
        const TProjectorField projectorField = TProjectorField::Parse("body.crop(100).crop(10)");

        UNIT_ASSERT_VALUES_EQUAL("body.crop(100).crop(10)", projectorField.GetRepresentation());

        UNIT_ASSERT_VALUES_EQUAL(TField::Body, projectorField.GetField());
        TVector<TString> words;
        for (int i = 0; i < 60; i++) {
            words.push_back(ToString(i));
        }

        TVector<TString> cropped;
        for (int i = 0; i < 10; i++) {
            cropped.push_back(ToString(i));
        }
        UNIT_ASSERT_VALUES_EQUAL(cropped, projectorField.Apply(words));
    }
    Y_UNIT_TEST(Empty) {
        const TProjectorField projectorField = TProjectorField::Parse("subject");

        UNIT_ASSERT_VALUES_EQUAL(TField::Subject, projectorField.GetField());
        TVector<TString> words;
        for (int i = 0; i < 60; i++) {
            words.push_back(ToString(i));
        }
        UNIT_ASSERT_VALUES_EQUAL(words, projectorField.Apply(words));
    }

    Y_UNIT_TEST(SegmentLenMore) {
        TVector<TString> words;

        for (int i = 1; i < 9; i++) {
            words.push_back(ToString(i));
        }

        TVector<TString> segments = {"{\"segments\":[\"1 2 3 4\",\"2 3 4 5\",\"3 4 5 6\",\"4 5 6 7\",\"5 6 7 8\"]}"};

        const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("segment(4,3,10)");
        UNIT_ASSERT_VALUES_EQUAL(dynamic_cast<const TSegmentProjector*>(p.Get())->GetOverlap(), 3);
        UNIT_ASSERT_EQUAL(p->Apply(words), segments);
    }

    Y_UNIT_TEST(SegmentLenLess) {
        TVector<TString> words;

        for (int i = 1; i < 9; i++) {
            words.push_back(ToString(i));
        }

        TVector<TString> segments = {"{\"segments\":[\"1 2 3 4\",\"2 3 4 5\",\"3 4 5 6\"]}"};

        const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("segment(4,3,6)");
        UNIT_ASSERT_VALUES_EQUAL(dynamic_cast<const TSegmentProjector*>(p.Get())->GetMaxLen(), 6);
        UNIT_ASSERT_EQUAL(p->Apply(words), segments);

    }

    Y_UNIT_TEST(SegmentEqual) {
        TVector<TString> words;

        for (int i = 1; i < 7; i++) {
            words.push_back(ToString(i));
        }

        TVector<TString> segments = {"{\"segments\":[\"1 2 3 4 5 6\"]}"};

        const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("segment(6,5,6)");
        UNIT_ASSERT_VALUES_EQUAL(dynamic_cast<const TSegmentProjector*>(p.Get())->GetMaxLen(), 6);
        UNIT_ASSERT_EQUAL(p->Apply(words), segments);

    }

    Y_UNIT_TEST(Plainify_empty) {
        TVector<TString> words = {""};
        TVector<TString> plainify = {""};

        const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("plainify_json_array");
        UNIT_ASSERT_EQUAL(p->Apply(words), plainify);

    }

    Y_UNIT_TEST(Plainify_with_elements) {
        TVector<TString> words = {"[\"1\", \"2\"]"};
        TVector<TString> plainify = {"1 2"};

        const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("plainify_json_array");
        UNIT_ASSERT_EQUAL(p->Apply(words), plainify);

    }

    Y_UNIT_TEST(Plainify_with_complex_elements) {
        TVector<TString> words = {"[[\"1\", \"2\"], [\"3\", \"4\"]]"};
        TVector<TString> plainify = {"1 2 3 4"};

        const TSimpleSharedPtr<const IProjector> p = ProjectorFactory("plainify_json_array");
        UNIT_ASSERT_EQUAL(p->Apply(words), plainify);

    }
}
