#include "gc.h"
#include "context.h"
#include "so2_answer.h"

namespace NLua {
    int TContext::TPtr::ActivateRule(TLuaStateHolder &state) const {
        Y_VERIFY(Base.RuleProvider);
        state.require(2);
        const TStringBuf ruleName = state.to_string(2);
        Base.RuleProvider->GetMutable(ruleName).Activate();
        return 0;
    }

    int TContext::TPtr::IsRuleActive(TLuaStateHolder &state) const {
        Y_VERIFY(Base.RuleProvider);
        state.require(2);
        const TStringBuf ruleName = state.to_string(2);
        state.push_bool(Base.RuleProvider->Get(ruleName).IsActive());
        return 1;
    }

    int TContext::TPtr::GetRule(TLuaStateHolder &state) const {
        Y_VERIFY(Base.RuleProvider);
        state.require(2);
        const TStringBuf ruleName = state.to_string(2);
        state.push_userdata(Base.RuleProvider->GetMutable(ruleName).AsPtr());
        return 1;
    }

    int TContext::TPtr::MlLog(TLuaStateHolder &state) const {
        Y_VERIFY(Base.MlLogProvider);
        state.require(3);
        Base.MlLogProvider->Log(state.to_string(2), state.to_string(3));
        return 0;
    }

    int TContext::TPtr::DlvLog(TLuaStateHolder &state) const {
        Y_VERIFY(Base.DlvLogProvider);
        state.require(3);
        Base.DlvLogProvider->Log(state.to_string(2), state.to_string(3));
        return 0;
    }

    int TContext::TPtr::GetSo2Context(TLuaStateHolder &state) const {
        state.require(1);
        state.create_table();
        if (Base.So2Answer) {
            state.push_userdata(TAnswer(*Base.So2Answer));
        } else {
            state.push_nil();
        }
        return 1;
    }

    int TContext::TPtr::AddType(TLuaStateHolder &state) const {
        Y_VERIFY(Base.RuleProvider);
        state.require(2);
        const TStringBuf type = state.to_string(2);
        Base.Types.emplace_back(type);
        return 0;
    }

    int TContext::TPtr::CheckShingle(TLuaStateHolder &state) const {
        Y_VERIFY(Base.ShinglerProvider);
        state.require(3);
        state.push_string(Base.ShinglerProvider->CheckShingle(state.to_number<ui32>(2), state.to_string(3)));
        return 1;
    }

    TContext::TPtr::TPtr(TContext &base) noexcept: Base(base) {}

    TContext::TContext(const NHtmlSanMisc::TAnswer *so2Answer,
                       IRuleProvider *ruleProvider,
                       IKVLogProvider *dlvLogProvider,
                       IKVLogProvider *mlLogProvider,
                       IShinglerProvider* shinglerProvider) noexcept
            : So2Answer(so2Answer)
            , RuleProvider(ruleProvider)
            , DlvLogProvider(dlvLogProvider)
            , MlLogProvider(mlLogProvider)
            , ShinglerProvider(shinglerProvider) {
    }

    TContext::TPtr TContext::AsPtr() &{
        return TPtr(*this);
    }

    void TContext::SetSo2Answer(const NHtmlSanMisc::TAnswer *so2Answer) {
        So2Answer = so2Answer;
    }

    void TContext::SetRuleProvider(IRuleProvider *ruleProvider) {
        RuleProvider = ruleProvider;
    }

    void TContext::SetDlvLogProvider(IKVLogProvider *logProvider) {
        DlvLogProvider = logProvider;
    }

    void TContext::SetMlLogProvider(IKVLogProvider *logProvider) {
        MlLogProvider = logProvider;
    }

    void TContext::SetShinglerProvider(IShinglerProvider *shinglerProvider) {
        ShinglerProvider = shinglerProvider;
    }

    const TVector<TString> &TContext::GetTypes() const {
        return Types;
    }

    const luaL_Reg TContext::TPtr::LUA_FUNCTIONS[] = {
            {"activate_rule",   NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::ActivateRule>},
            {"is_rule_active",  NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::IsRuleActive>},
            {"get_rule",        NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::GetRule>},
            {"dlv_log",         NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::DlvLog>},
            {"ml_log",          NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::MlLog>},
            {"get_so2_context", NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::GetSo2Context>},
            {"add_type",        NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::AddType>},
            {"check_shingle",   NLua::MethodConstHandler<TContext::TPtr, &TContext::TPtr::CheckShingle>},
            {GC,                NLua::Destructor<TContext::TPtr>},
            {nullptr,           nullptr}};

    const char TContext::TPtr::LUA_METATABLE_NAME[] = "context_mt";
}
