#pragma once

#include <util/generic/strbuf.h>
#include <util/system/protect.h>
#include "kv_logger.h"
#include "context.h"

namespace NRulesDummyContext {
    class TRule : public NLua::IRule {
    public:
        TRule() = default;

        void Activate() override {
            Active = true;
        }

        [[nodiscard]] bool IsActive() const override {
            return Active;
        }

        void SetScore(double score) override {
            Score = score;
        }

        [[nodiscard]] double GetScore() const override {
            return Score;
        }

    private:
        bool Active{};
        double Score{};
    };

    class TRuleProvider : public NLua::IRuleProvider {
    public:
        const NLua::IRule &Get(const TStringBuf &ruleName) const override {
            return Rules.at(ruleName);
        }

        [[nodiscard]] NLua::IRule &GetMutable(const TStringBuf &ruleName) override {
            return Rules[ruleName];
        };

    private:
        THashMap<TString, TRule> Rules;
    };

    class TLogProvider : public NLua::IKVLogProvider {
    public:
        void Log(const TStringBuf &key, const TStringBuf &value) override {
            LogValues.emplace(key, value);
        }

        [[nodiscard]] const THashMap<TString, TString> &GetLogValues() const {
            return LogValues;
        }

    private:
        THashMap<TString, TString> LogValues;
    };

    class TShinglerProvider : public NLua::IShinglerProvider {
    public:
        [[nodiscard]] const THashSet<TString> &Get(ui32 shingleType) const {
            return ShinglesByType.at(shingleType);
        }

        TString CheckShingle(ui32 shingleType, const TStringBuf &text) final {
            return *ShinglesByType[shingleType].emplace(text).first;
        }

    private:
        THashMap<ui32, THashSet<TString>> ShinglesByType;
    };

    class TContext : public NLua::TContext {
    public:
        const TLogProvider &GetDlvLog() const {
            return DlvLog;
        }

        const TLogProvider &GetMlLog() const {
            return MlLog;
        }

        const TRuleProvider &GetRuleProvider() const {
            return RuleProvider;
        }

        const TShinglerProvider &GetShingler() const {
            return Shingler;
        }

        TContext(const NHtmlSanMisc::TAnswer &so2Answer) {
            SetSo2Answer(&so2Answer);
            SetRuleProvider(&RuleProvider);
            SetDlvLogProvider(&DlvLog);
            SetMlLogProvider(&MlLog);
            SetShinglerProvider(&Shingler);
        }

    private:
        TLogProvider DlvLog;
        TLogProvider MlLog;
        TRuleProvider RuleProvider;
        TShinglerProvider Shingler;
    };
};