#include <library/cpp/json/json_value.h>
#include <library/cpp/lua/json.h>
#include <util/stream/str.h>
#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/generic/scope.h>
#include <util/folder/path.h>
#include <util/stream/buffer.h>
#include <util/stream/tee.h>
#include "pcre.h"
#include "runner.h"
#include "print.h"

namespace NLua {
    TRunner::TRunner() {
        LuaStateHolder.BootStrap();
        NLua::BootstrapPcre(LuaStateHolder);
        NLua::BootstrapPrint(LuaStateHolder);
    }

    void TRunner::Run(TContext::TPtr context, const TZtStringBuf &function) {
        LuaStateHolder.push_global(function.c_str());
        LuaStateHolder.push_userdata(context);
        LuaStateHolder.call(1, 0);
    }

    void TRunner::Run(TContext::TPtr context, const TZtStringBuf &function, const TVector<NJson::TJsonValue> &values) {
        LuaStateHolder.push_global(function.c_str());
        LuaStateHolder.push_userdata(context);
        for (const NJson::TJsonValue &value: values) {
            NLua::PushJsonValue(&LuaStateHolder, value);
        }
        LuaStateHolder.call(1 + values.size(), 0);
    }

    bool TRunner::FunctionExists(const TZtStringBuf &function) {
        LuaStateHolder.push_string(function.c_str());
        LuaStateHolder.push_global("_G");
        return !LuaStateHolder.pop_nil();
    }

    void TRunner::Load(const TFsPath &entry) {
        const TString cwd = NFs::CurrentWorkingDirectory();
        Y_DEFER {
                    NFs::SetCurrentWorkingDirectory(cwd);
                };
        NFs::SetCurrentWorkingDirectory(entry.Parent());

        Load(TIFStream(entry), entry.c_str());
    }

    void TRunner::Load(IInputStream &stream, const TZtStringBuf &tag) {
        TStringStream current;
        {
            TBufferOutput allScriptStream(AllScripts.emplace_back(tag, TBuffer{}).second);
            TTeeOutput tee(&allScriptStream, &current);
            TransferData(&stream, &tee);
        }

        LuaStateHolder.Load(&current, tag.c_str());
        LuaStateHolder.call(0, 0);
    }

    void TRunner::Load(IInputStream &&stream, const TZtStringBuf &tag) {
        Load(stream, tag);
    }

    TRunner TRunner::Clone() const {
        TRunner newRunner;
        for (const auto&[tag, script]: AllScripts) {
            newRunner.Load(TBufferInput(script), tag);
        }
        return newRunner;
    }
}
