#pragma once

#include <util/generic/ptr.h>
#include <util/system/mutex.h>
#include <util/system/condvar.h>
#include <util/system/spinlock.h>
#include <mail/so/libs/talkative_config/config.h>
#include <library/cpp/coroutine/engine/impl.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/thread/lfqueue.h>
#include <util/thread/lfstack.h>

struct TPoolParams {
    size_t poolSize = 32;
    TDuration timeout = TDuration::MilliSeconds(100);

    TPoolParams() = default;

    explicit TPoolParams(const NConfig::TConfig & config) {
        if(config.Has("size")) {
            poolSize = NTalkativeConfig::As<size_t>(config, "size");
        }
        if(config.Has("timeout")) {
            timeout = NTalkativeConfig::As<TDuration>(config, "timeout");
        }
    }
    TPoolParams(size_t poolSize, const TDuration& timeout) noexcept
            : poolSize(poolSize)
            , timeout(timeout)
    {
    }
};

template <class TPoolItem, template<class ...> class TQueue = TLockFreeStack>
class TPool {
    struct TCountedQueue{
        explicit TCountedQueue(size_t poolSize) : poolSize(poolSize) {}
        ~TCountedQueue() {
            TPoolItem* item;
            while (stack.Dequeue(&item)) delete item;
        }

        TAtomic created{};
        TQueue<TPoolItem*> stack;
        TMutex waitMutex;
        TCondVar condVar;
        size_t poolSize;
    };
public:
    struct TItemHolder {
    public:
        void Reset() {
            Destroy(true);
        }

        bool Defined() const {
            return item && masterPool;
        }

        TPoolItem& Get() const { return *item; };
        TPoolItem* operator->() { return item.Get(); };
        const TPoolItem* operator->() const { return item.Get(); };
        TPoolItem& operator*() { return *item; };
        const TPoolItem& operator*() const { return *item; };

        TItemHolder() = default;
        TItemHolder(TItemHolder&&) noexcept = default;
        TItemHolder &operator=(TItemHolder&& another) noexcept {
            Destroy(false);
            item = std::move(another.item);
            masterPool = std::move(another.masterPool);
            return *this;
        };

        ~TItemHolder() {
            Destroy(false);
        }

    private:
        friend class TPool;
        TItemHolder(THolder<TPoolItem> && item, TCountedQueue& masterPool)
                : item(std::move(item)), masterPool(&masterPool)
        {}

        void Destroy(bool unlink) {
            if (!Defined())
                return;
            if (!unlink) {
                masterPool->stack.Enqueue(item.Release());
            } else {
                AtomicDecrement(masterPool->created);
            }
            masterPool->condVar.Signal();
            item.Reset();
            masterPool.Reset();
        }

    private:
        THolder<TPoolItem> item;
        THolder<TCountedQueue, TNoAction> masterPool{};
    };

    class TPoolItemTraits {
    public:
        virtual THolder<TPoolItem> create() const = 0;
        virtual ~TPoolItemTraits() = default;
    };

    const TPoolParams& getParams() const {
        return params;
    }

    bool get(const TPoolItemTraits& traits, TItemHolder& res, TInstant until, TCont * cont = nullptr) {
        while(true) {
            {
                TPoolItem* item{};
                if(countedQueue.stack.Dequeue(&item)) {
                    res = {THolder<TPoolItem>(item), countedQueue};
                    return true;
                }
            }

            if(Now() > until)
                return false;

            if (AtomicGetAndAdd(countedQueue.created, 1) < (intptr_t) params.poolSize) {
                THolder<TPoolItem> item = traits.create();
                res = {std::move(item), countedQueue};
                return true;
            }

            AtomicDecrement(countedQueue.created);

            if (cont != nullptr) {
                cont->SleepT(TDuration::MilliSeconds(5));
            } else with_lock(countedQueue.waitMutex) {
                countedQueue.condVar.WaitT(countedQueue.waitMutex, TDuration::MilliSeconds(5));
            }
        }
    }

    bool get(const TPoolItemTraits& traits, TItemHolder& res, TCont * cont = nullptr) { return get(traits, res, params.timeout, cont); }
    bool get(const TPoolItemTraits& traits, TItemHolder& res, const TDuration& timeout, TCont * cont = nullptr) { return get(traits, res, timeout.ToDeadLine(), cont); }

    TItemHolder get(const TPoolItemTraits& traits, const TInstant& until, TCont * cont = nullptr) {
        TItemHolder item;
        if(!get(traits, item, until, cont))
            ythrow TWithBackTrace<yexception>() << "pool timeout";
        return item;
    }

    TItemHolder get(const TPoolItemTraits& traits, const TDuration& timeout, TCont * cont = nullptr) { return get(traits, timeout.ToDeadLine(), cont); }
    TItemHolder get(const TPoolItemTraits& traits, TCont * cont = nullptr) { return get(traits, params.timeout.ToDeadLine(), cont); }

    explicit TPool(const TPoolParams & params = {}) : params(params), countedQueue(params.poolSize) {}

private:
    TPoolParams params;
    TCountedQueue countedQueue;
};

template <class T>
using TFifoPool = TPool<T, TLockFreeStack>;

template <class T>
using TLifoPool = TPool<T, TLockFreeQueue>;
