#pragma once

#include <util/system/protect.h>

#include <array>

template <class T>
struct TStdArraySize;

template <class T, size_t N>
struct TStdArraySize<std::array<T, N>> {
    static constexpr size_t size = N;
};

template <class T>
class TTrueConst {
private:
    constexpr static size_t MaxPageSize = 8192;
    constexpr static size_t AlignUp = (sizeof(T) + MaxPageSize - 1) & ~(MaxPageSize - 1);
    alignas(MaxPageSize) char Mem[AlignUp];

public:
    using value_type = T;

public:
    template<class... Args>
    TTrueConst(Args&&... args) {
        new (Data()) T{std::forward<Args>(args)...};
        ProtectMemory(Mem, sizeof Mem, PM_READ);
    }

    ~TTrueConst() {
        ProtectMemory(Mem, sizeof Mem, PM_READ | PM_WRITE);
        Data()->~T();
    }

public:
    constexpr const T* Data() const noexcept {
        return reinterpret_cast<const T*>(Mem);
    }

    constexpr T* Data() noexcept {
        return reinterpret_cast<T*>(Mem);
    }

    constexpr const T& operator*() const& noexcept {
        return *Data();
    }

    constexpr const T* operator->() const noexcept {
        return Data();
    }

    template <class Key>
    constexpr const auto& operator[](Key key) const noexcept(noexcept((*((T*)nullptr))[Key{}])) {
        return (*Data())[key];
    }
};

template <class T>
constexpr auto MakeTrueConst(T&& arg) {
    return TTrueConst<T>{std::move(arg)};
}

template <class... Args>
constexpr auto MakeTrueConstArray(Args&&... args) {
    using CT = std::common_type_t<Args...>;
    using T = std::array<CT, sizeof...(Args)>;
    return TTrueConst<T>{T{std::forward<CT>(args)...}};
}

