#include <utility>

#include "scheduler.h"

struct TSimpleScheduler::TTaskInfo {
    TTaskInfo() {}
    TTaskInfo(const TTask& task, const TDuration& interval, const TString& name, const TAtomicSharedPtr<IProgressInfo>& info)
        : task(task), name(name), interval(interval), info(info)
    {
        Y_VERIFY(info);
    }

    TTaskInfo(TTaskInfo&&) = default;
    TTaskInfo(const TTaskInfo&) = default;

    TTaskInfo& operator=(TTaskInfo&&) = default;
    TTaskInfo& operator=(const TTaskInfo&) = default;

    void Run() const
    try {
        info->OnStart(name);
        task();
        info->OnFinish(name);
    } catch (...) {
        info->OnException(name);
    }

    void Clear() {
        info.Drop();
        task = TTask();
    }

    TTask task;
    TString name;
    TDuration interval;
    TAtomicSharedPtr<IProgressInfo> info;
};

struct TEmptyProgressInfo : public TSimpleScheduler::IProgressInfo {
    void OnStart(const TString&) final { }
    void OnFinish(const TString&) final { }
    void OnException(const TString&) final { }
};

TSimpleScheduler::TSimpleScheduler(IThreadFactory& pool, const TAtomicSharedPtr<IProgressInfo>& default_progress_info)
    : m_pool(pool)
    , m_default_progress_info(default_progress_info)
{
    if (!m_default_progress_info)
        m_default_progress_info = MakeAtomicShared<TEmptyProgressInfo>();
    m_pool.Run(this).Swap(m_thread);
}

TSimpleScheduler::~TSimpleScheduler() {
    with_lock(m_mutex) {
        m_running = false;
        m_condvar.Signal();
    }
    m_thread->Join();
}

void TSimpleScheduler::Add(TTask task, const TDuration& interval, const TString& name, const TAtomicSharedPtr<IProgressInfo>& progress_info) {
    Add(std::move(task), TInstant::Now() + interval, interval, name, progress_info);
}

void TSimpleScheduler::Add(TTask task, const TInstant& first_start, const TDuration& interval, const TString& name, const TAtomicSharedPtr<IProgressInfo>& progress_info) {
    with_lock(m_mutex) {
        auto it = m_tasks.emplace(first_start, TTaskInfo(task, interval, name, progress_info ? progress_info : m_default_progress_info));
        if (m_tasks.begin() == it)
            m_condvar.Signal();
    }
}

bool TSimpleScheduler::WaitNextTask() {
    return m_tasks.empty() ?
        m_condvar.WaitI(m_mutex), false:
        !m_condvar.WaitD(m_mutex, m_tasks.begin()->first);
}

bool TSimpleScheduler::Get(TTaskInfo& task) {
    if (!WaitNextTask())
        return false;

    auto it = m_tasks.begin();
    task = std::move(it->second);
    m_tasks.erase(it);

    return true;
}

void TSimpleScheduler::ExecuteTask(const TTaskInfo& task)
try {
    m_pool.Run([task]() { task.Run(); });
}
catch (...) {
    task.info->OnException(task.name);
}

void TSimpleScheduler::DoExecute() {
    TTaskInfo task;
    with_lock(m_mutex) {
        while (m_running) {
            if (Get(task)) {
                m_tasks.emplace(TInstant::Now() + task.interval, task);

                TInverseGuard<TMutex> unlock(m_mutex);
                ExecuteTask(task);
                task.Clear();
            }
        }
    }
}

TScheduler::TScheduler(IThreadFactory& pool, const TAtomicSharedPtr<IProgressInfo>& default_progress_info)
    : TSimpleScheduler(pool, default_progress_info)
{ }

void TScheduler::Remove(const TString& name) {
    with_lock(m_mutex) {
        for (auto it = m_tasks.begin(); it != m_tasks.end(); ) {
            if (it->second.name == name)
                m_tasks.erase(it++);
            else
                ++it;
        }
    }
}

bool TScheduler::WaitNextTask() {
    return TSimpleScheduler::WaitNextTask() && !m_tasks.empty() && m_tasks.begin()->first <= TInstant::Now();
}
