#pragma once

#include <functional>
#include <util/generic/map.h>
#include <util/thread/factory.h>
#include <util/system/condvar.h>
#include <util/datetime/base.h>
#include <library/cpp/logger/log.h>

class TSimpleScheduler : private IThreadFactory::IThreadAble {
public:
    struct IProgressInfo {
        virtual void OnStart(const TString&) = 0;
        virtual void OnFinish(const TString&) = 0;
        virtual void OnException(const TString&) = 0;
        virtual ~IProgressInfo() = default;
    };

    typedef std::function<void()> TTask;

    explicit TSimpleScheduler(IThreadFactory& pool, const TAtomicSharedPtr<IProgressInfo>& default_progress_info = nullptr);
    ~TSimpleScheduler() override;

    void Add(TTask task, const TDuration& interval, const TString& name = nullptr, const TAtomicSharedPtr<IProgressInfo>& progress_info = nullptr);
    void Add(TTask task, const TInstant& first_start, const TDuration& interval, const TString& name = nullptr, const TAtomicSharedPtr<IProgressInfo>& progress_info = nullptr);
protected:
    virtual bool WaitNextTask();

    TMutex m_mutex;
    TCondVar m_condvar;

    struct TTaskInfo;
    TMultiMap<TInstant, TTaskInfo> m_tasks;
private:
    void DoExecute() final;
    bool Get(TTaskInfo& task);
    void ExecuteTask(const TTaskInfo& task);

    IThreadFactory& m_pool;
    THolder<IThreadFactory::IThread> m_thread;
    TAtomicSharedPtr<IProgressInfo> m_default_progress_info;
    bool m_running = true;
};

class TScheduler : public TSimpleScheduler {
public:
    explicit TScheduler(IThreadFactory& pool, const TAtomicSharedPtr<IProgressInfo>& default_progress_info = nullptr);

    void Remove(const TString& name);
private:
    bool WaitNextTask() final;
};
