#include "scheduler.h"
#include <util/thread/pool.h>
#include <library/cpp/testing/unittest/registar.h>
#include <atomic>

Y_UNIT_TEST_SUITE(Sheduler) {
    Y_UNIT_TEST(Wait) {
        TAdaptiveThreadPool threadPool;
        threadPool.Start(0, 0);
        std::atomic_bool now(false), later(false);

        {TSimpleScheduler scheduler(threadPool);
            scheduler.Add([&now]() { now.store(true); }, TInstant::Now(), TDuration::Days(1));
            scheduler.Add([&later]() { later.store(true); }, TDuration::Days(1));
            sleep(1);
        }

        UNIT_ASSERT_EQUAL(now, true);
        UNIT_ASSERT_EQUAL(later, false);
    }

    Y_UNIT_TEST(Count) {
        TAdaptiveThreadPool threadPool;
        threadPool.Start(2, 0);
        std::atomic_uint32_t one(0), two(0), three(0);

        {TSimpleScheduler scheduler(threadPool);
            scheduler.Add([&three]() { ++three; }, TDuration::MilliSeconds(300));
            scheduler.Add([&two]() { ++two; }, TDuration::MilliSeconds(200));
            scheduler.Add([&one]() { ++one; }, TDuration::MilliSeconds(100));
            sleep(1);
        }

        UNIT_ASSERT_EQUAL(one > two, true);
        UNIT_ASSERT_EQUAL(two > three, true);
        UNIT_ASSERT_EQUAL(three >= 2, true);
    }

    Y_UNIT_TEST(Remove) {
        TAdaptiveThreadPool threadPool;
        threadPool.Start(2, 0);
        std::atomic_uint32_t one(0), two(0), three(0);

        {TScheduler scheduler(threadPool);
            scheduler.Add([&one]() { ++one; }, TDuration::MilliSeconds(50), "one");
            scheduler.Add([&two]() { ++two; }, TDuration::MilliSeconds(50), "two");
            scheduler.Add([&three]() { ++three; }, TDuration::MilliSeconds(50), "three");

            usleep(100 * 1000);
            scheduler.Remove("one");
            usleep(100 * 1000);
            scheduler.Remove("two");
            usleep(100 * 1000);
            scheduler.Remove("three");
            usleep(100 * 1000);
        }

        UNIT_ASSERT_EQUAL(three > two, true);
        UNIT_ASSERT_EQUAL(two > one, true);
        UNIT_ASSERT_EQUAL(one > 0, true);
    }

    struct TProgress : public TSimpleScheduler::IProgressInfo {
        void OnStart(const TString&) override {
            started = true;
        }

        void OnFinish(const TString&) override {
            finished = true;
        }

        void OnException(const TString&) override {
            try{throw;}
            catch(const std::exception& e) {
                exception = e.what();
            }
        }

        bool started = false;
        bool finished = false;
        TString exception;
     };

    Y_UNIT_TEST(Informer) {
        TAdaptiveThreadPool threadPool;
        threadPool.Start(0, 0);
        TAtomicSharedPtr<TProgress> one(new TProgress), two(new TProgress);

        {TSimpleScheduler scheduler(threadPool);
            scheduler.Add([]() { }, TDuration::Seconds(0.5), "one", one);
            scheduler.Add([]() { throw std::runtime_error("Test"); }, TDuration::Seconds(0.5), "two", two);
            sleep(1);
        }

        UNIT_ASSERT_EQUAL(one->started, true);
        UNIT_ASSERT_EQUAL(one->finished, true);
        UNIT_ASSERT_EQUAL(one->exception.empty(), true);

        UNIT_ASSERT_EQUAL(two->started, true);
        UNIT_ASSERT_EQUAL(two->finished, false);
        UNIT_ASSERT_EQUAL(two->exception, "Test");
    }
}
