#pragma once

#include <library/cpp/config/config.h>
#include <util/folder/path.h>
#include <util/datetime/base.h>
#include <util/system/type_name.h>

Y_DECLARE_OUT_SPEC(inline, NConfig::TConfig, stream, config) {
    config.DumpJson(stream);
}

namespace NTalkativeConfig {
    const NConfig::TConfig& Get(const NConfig::TConfig& config, const TStringBuf& name);
    const NConfig::TConfig* Find(const NConfig::TConfig& config, const TStringBuf& name);

    template <class T>
    const T& Get(const NConfig::TConfig& config, const TStringBuf& name = nullptr) {
        const auto& localConf = name ? Get(config, name) : config;

        if (!localConf.IsA<T>())
            ythrow TWithBackTrace<yexception>() << name << " field must be a " << TypeName<T>() << " : " << localConf << "; from " << config;

        return localConf.Get<T>();
    }

    template <class T>
    const T * Find(const NConfig::TConfig& config, const TStringBuf& name = nullptr) {
        if(!name)
            return &config.Get<T>();
        else if (const auto localConf = Find(config, name))
            return &localConf->Get<T>();
        return nullptr;
    }

    template <class T>
    T As(const NConfig::TConfig& config, const TStringBuf& name = nullptr) {
        const auto& localConf = name ? Get(config, name) : config;
        return localConf.As<T>();
    }
    template <>
    inline TFsPath As<TFsPath>(const NConfig::TConfig& config, const TStringBuf& name) {
        TFsPath path = Get<TString>(config, name);

        if (!path.Exists())
            ythrow TWithBackTrace<yexception>() << "path " << path << " from config " << config << " doesn't exist";

        return path;
    }

    template <>
    inline TInstant As<TInstant>(const NConfig::TConfig& config, const TStringBuf& name) {
        const auto& strTime = Get<TString>(config, name);
        if (TInstant result; TInstant::TryParseIso8601(strTime, result))
            return result;
        ythrow TWithBackTrace<yexception>() << "Cannot parse date from " << strTime << " from config " << config;
    }

    template <>
    inline TDuration As<TDuration>(const NConfig::TConfig& config, const TStringBuf& name) {
        const auto& strTime = Get<TString>(config, name);
        if (TDuration result; TDuration::TryParse(strTime, result))
            return result;
        ythrow TWithBackTrace<yexception>() << "Cannot parse duration from " << strTime << " from config " << config;
    }
} // namespace NTalkativeConfig
