#include <library/cpp/archive/yarchive.h>
#include <util/generic/string.h>
#include <util/generic/hash_set.h>
#include <util/stream/input.h>
#include <util/memory/blob.h>
#include <util/string/reverse.h>
#include <util/string/strip.h>
#include "is_tld.h"

namespace NTld {
    static const unsigned char resource[] = {
#include "data.inc"
    };

    TVector<TString> ReadTlds(IInputStream *stream) {
        Y_VERIFY(stream);

        TString line;

        TVector<TString> tlds;

        while (stream->ReadLine(line)) {
            const auto view = StripString(TStringBuf(line), [](const char* c){
                return *c == '.' || IsAsciiSpace(*c);
            });

            if (!view || view.StartsWith("//")) {
                continue;
            }

            tlds.emplace_back(view);
        }

        return tlds;
    }

    TDetector::TDetector() : TDetector(ReadTlds(TArchiveReader(TBlob::NoCopy(resource, sizeof(resource))).ObjectByKey("/tld.txt").Get())) {

    }

    TDetector::TDetector(const TVector<TString>& tlds) {
        TTrieBuilder builder;

        for(auto tld: tlds) {
            ReverseInPlace(tld);
            builder.Add(tld + ".", true);
        }

        TBufferStream buf;
        builder.Save(buf);

        Trie.Init(TBlob::FromBuffer(buf.Buffer()));

        Y_VERIFY(Trie.IsInitialized());
    }

    size_t TDetector::FindLongestTldSuffixSize(TString host) const {
        ReverseInPlace(host);
        if(size_t len{}; Trie.FindLongestPrefix(host, &len)) {
            return len - 1;
        }
        return NPOS;
    }

    bool TDetector::IsTld(TString host) const {
        ReverseInPlace(host);
        return Trie.Find(host + ".");
    }
}
